import { Express }        from 'express-serve-static-core';
import express            from 'express';
import { StatusCodes }    from 'http-status-codes';
import ExerciseManager    from '../managers/ExerciseManager';
import AssignmentManager  from '../managers/AssignmentManager';
import TagManager         from '../managers/TagManager';
import TagProposalManager from '../managers/TagProposalManager';
import UserManager        from '../managers/UserManager';


type GetFunction = (id: string | number, ...args: Array<unknown>) => Promise<unknown>


class ParamsCallbackManager {
    protected listenParam(paramName: string, backend: Express, getFunction: GetFunction, args: Array<unknown>, indexName: string) {
        backend.param(paramName, (req: express.Request, res: express.Response, next: express.NextFunction, id: string | number) => {
            getFunction(id, ...args).then(result => {
                if ( result ) {
                    this.initBoundParams(req);
                    (req.boundParams as Record<string, unknown>)[indexName] = result;

                    next();
                } else {
                    req.session.sendResponse(res, StatusCodes.NOT_FOUND, {}, 'Param bounding failed: ' + paramName);
                }
            });
        });
    }

    initBoundParams(req: express.Request) {
        if ( !req.boundParams ) {
            req.boundParams = {
                user       : undefined,
                assignment : undefined,
                exercise   : undefined,
                tag        : undefined,
                tagProposal: undefined
            };
        }
    }

    registerOnBackend(backend: Express) {
        this.listenParam('userId', backend, (UserManager.getById as GetFunction).bind(UserManager), [ {
            assignments: true,
            exercises  : {
                include: {
                    members   : true,
                    assignment: {
                        include: {
                            staff: true
                        }
                    }
                }
            }
        } ], 'user');

        this.listenParam('assignmentNameOrUrl', backend, (AssignmentManager.get as GetFunction).bind(AssignmentManager), [ {
            exercises: true,
            staff    : true
        } ], 'assignment');

        this.listenParam('exerciseIdOrUrl', backend, (ExerciseManager.get as GetFunction).bind(ExerciseManager), [ {
            assignment: {
                include: {
                    staff: true
                }
            },
            members   : true,
            results   : true
        } ], 'exercise');

        this.listenParam('tagName', backend, (TagManager.get as GetFunction).bind(TagManager), [ {
            assignments: true
        } ], 'tag');

        this.listenParam('tagProposalName', backend, (TagProposalManager.get as GetFunction).bind(TagProposalManager), [ {} ], 'tagProposal');
    }
}


export default new ParamsCallbackManager();
