import GitlabUser    from '../shared/types/Gitlab/GitlabUser';
import { Prisma }    from '@prisma/client';
import db            from '../helpers/DatabaseHelper';
import GitlabProfile from '../shared/types/Gitlab/GitlabProfile';
import { User }      from '../types/DatabaseTypes';


class UserManager {
    async getByMail(mail: string, include: Prisma.UserInclude | undefined = undefined): Promise<User | undefined> {
        return await db.user.findUnique({
                                            where  : {
                                                mail: mail
                                            },
                                            include: include
                                        }) as unknown as User ?? undefined;
    }

    async getById(id: number, include: Prisma.UserInclude | undefined = undefined): Promise<User | undefined> {
        return await db.user.findUnique({
                                            where  : {
                                                id: id
                                            },
                                            include: include
                                        }) as unknown as User ?? undefined;
    }

    async getUpdateFromGitlabProfile(gitlabProfile: GitlabProfile): Promise<User> {
        await db.user.upsert({
                                 where : {
                                     id: gitlabProfile.id
                                 },
                                 update: {
                                     mail          : gitlabProfile.email,
                                     gitlabLastInfo: gitlabProfile
                                 },
                                 create: {
                                     id            : gitlabProfile.id,
                                     name          : gitlabProfile.name,
                                     mail          : gitlabProfile.email,
                                     gitlabUsername: gitlabProfile.username,
                                     gitlabLastInfo: gitlabProfile,
                                     deleted       : false
                                 }
                             });

        return (await this.getById(gitlabProfile.id))!;
    }

    async getFromGitlabUser(gitlabUser: GitlabUser, createIfNotExist: boolean = false, include: Prisma.UserInclude | undefined = undefined): Promise<User | number | undefined> {
        let user = await this.getById(gitlabUser.id, include) ?? gitlabUser.id;

        if ( typeof user === 'number' && createIfNotExist ) {
            user = (await db.user.create({
                                             data: {
                                                 id            : gitlabUser.id,
                                                 gitlabUsername: gitlabUser.name
                                             }
                                         })).id;
        }

        return user;
    }

    async getFromGitlabUsers(gitlabUsers: Array<GitlabUser>, createIfNotExist: boolean = false, include: Prisma.UserInclude | undefined = undefined): Promise<Array<User | number | undefined>> {
        return Promise.all(gitlabUsers.map(gitlabUser => this.getFromGitlabUser(gitlabUser, createIfNotExist, include)));
    }
}


export default new UserManager();
