import { Express }               from 'express-serve-static-core';
import express                   from 'express';
import * as ExpressValidator     from 'express-validator';
import { StatusCodes }           from 'http-status-codes';
import RoutesManager             from '../express/RoutesManager';
import ParamsValidatorMiddleware from '../middlewares/ParamsValidatorMiddleware';
import SecurityMiddleware        from '../middlewares/SecurityMiddleware';
import GitlabUser                from '../shared/types/Gitlab/GitlabUser';
import GitlabManager             from '../managers/GitlabManager';
import Config                    from '../config/Config';
import GitlabRepository          from '../shared/types/Gitlab/GitlabRepository';
import { AxiosError }            from 'axios';
import logger                    from '../shared/logging/WinstonLogger';
import DojoValidators            from '../helpers/DojoValidators';
import { v4 as uuidv4 }          from 'uuid';
import GitlabMember              from '../shared/types/Gitlab/GitlabMember';
import GitlabAccessLevel         from '../shared/types/Gitlab/GitlabAccessLevel';
import { Prisma }                from '@prisma/client';
import { Assignment, Exercise }  from '../types/DatabaseTypes';
import db                        from '../helpers/DatabaseHelper';
import SecurityCheckType         from '../types/SecurityCheckType';
import GitlabTreeFile            from '../shared/types/Gitlab/GitlabTreeFile';
import GitlabFile                from '../shared/types/Gitlab/GitlabFile';
import GitlabTreeFileType        from '../shared/types/Gitlab/GitlabTreeFileType';
import JSON5                     from 'json5';
import fs                        from 'fs';
import path                      from 'path';
import AssignmentFile            from '../shared/types/Dojo/AssignmentFile';
import ExerciseResultsFile       from '../shared/types/Dojo/ExerciseResultsFile';
import DojoStatusCode            from '../shared/types/Dojo/DojoStatusCode';
import GlobalHelper              from '../helpers/GlobalHelper';
import { IFileDirStat }          from '../shared/helpers/recursiveFilesStats/RecursiveFilesStats';
import ExerciseManager           from '../managers/ExerciseManager';


class ExerciseRoutes implements RoutesManager {
    private readonly exerciseValidator: ExpressValidator.Schema = {
        members: {
            trim           : true,
            notEmpty       : true,
            customSanitizer: DojoValidators.jsonSanitizer
        }
    };

    private readonly resultValidator: ExpressValidator.Schema = {
        exitCode     : {
            isInt: true,
            toInt: true
        },
        commit       : {
            trim           : true,
            notEmpty       : true,
            customSanitizer: DojoValidators.jsonSanitizer
        },
        results      : {
            trim           : true,
            notEmpty       : true,
            custom         : DojoValidators.exerciseResultsValidator,
            customSanitizer: DojoValidators.jsonSanitizer
        },
        files        : {
            trim           : true,
            notEmpty       : true,
            customSanitizer: DojoValidators.jsonSanitizer
        },
        archiveBase64: {
            isBase64: true,
            notEmpty: true
        }
    };

    registerOnBackend(backend: Express) {
        backend.post('/assignments/:assignmentNameOrUrl/exercises', SecurityMiddleware.check(true, SecurityCheckType.ASSIGNMENT_IS_PUBLISHED), ParamsValidatorMiddleware.validate(this.exerciseValidator), this.createExercise.bind(this));

        backend.get('/exercises/:exerciseIdOrUrl/assignment', SecurityMiddleware.check(false, SecurityCheckType.EXERCISE_SECRET), this.getAssignment.bind(this));

        backend.post('/exercises/:exerciseIdOrUrl/results', SecurityMiddleware.check(false, SecurityCheckType.EXERCISE_SECRET), ParamsValidatorMiddleware.validate(this.resultValidator), this.createResult.bind(this));
    }

    private getExerciseName(assignment: Assignment, members: Array<GitlabUser>, suffix: number): string {
        const memberNames: string = members.map(member => member.username).sort((a, b) => a.localeCompare(b)).join(' + ');
        const suffixString: string = suffix > 0 ? ` - ${ suffix }` : '';
        return `DojoEx - ${ assignment.name } - ${ memberNames }${ suffixString }`;
    }

    private getExercisePath(assignment: Assignment, exerciseId: string): string {
        return `dojo-ex_${ (assignment.gitlabLastInfo as unknown as GitlabRepository).path }_${ exerciseId }`;
    }

    private async checkExerciseLimit(assignment: Assignment, members: Array<GitlabUser>): Promise<Array<GitlabUser>> {
        const exercises: Array<Exercise> | undefined = await ExerciseManager.getFromAssignment(assignment.name, { members: true });
        const reachedLimitUsers: Array<GitlabUser> = [];
        if ( exercises ) {
            for ( const member of members ) {
                const exerciseCount: number = exercises.filter(exercise => exercise.members.findIndex(exerciseMember => exerciseMember.id === member.id) !== -1).length;
                if ( exerciseCount >= Config.exercise.maxPerAssignment ) {
                    reachedLimitUsers.push(member);
                }
            }
        }

        return reachedLimitUsers;
    }

    private async createExerciseRepository(assignment: Assignment, members: Array<GitlabUser>, exerciseId: string, req: express.Request, res: express.Response): Promise<GitlabRepository | undefined> {
        let repository!: GitlabRepository;

        let suffix: number = 0;
        do {
            try {
                repository = await GitlabManager.forkRepository((assignment.gitlabCreationInfo as unknown as GitlabRepository).id, this.getExerciseName(assignment, members, suffix), this.getExercisePath(req.boundParams.assignment!, exerciseId), Config.exercise.default.description.replace('{{ASSIGNMENT_NAME}}', assignment.name), Config.exercise.default.visibility, Config.gitlab.group.exercises);
                break;
            } catch ( error ) {
                logger.error('Repo creation error');
                logger.error(error);

                if ( error instanceof AxiosError ) {
                    if ( error.response?.data.message.name && error.response.data.message.name === 'has already been taken' ) {
                        suffix++;
                    } else {
                        req.session.sendResponse(res, StatusCodes.INTERNAL_SERVER_ERROR, {}, 'Unknown gitlab error while forking repository', DojoStatusCode.EXERCISE_CREATION_GITLAB_ERROR);
                        return;
                    }
                } else {
                    req.session.sendResponse(res, StatusCodes.INTERNAL_SERVER_ERROR, {}, 'Unknown error while forking repository', DojoStatusCode.EXERCISE_CREATION_INTERNAL_ERROR);
                    return;
                }
            }
        } while ( suffix < Config.exercise.maxSameName );

        if ( suffix >= Config.exercise.maxSameName ) {
            logger.error('Max exercise with same name reached');
            req.session.sendResponse(res, StatusCodes.INSUFFICIENT_SPACE_ON_RESOURCE, undefined, 'Max exercise per assignment reached', DojoStatusCode.MAX_EXERCISE_PER_ASSIGNMENT_REACHED);
            return;
        }

        return repository;
    }

    private async createExercise(req: express.Request, res: express.Response) {
        const params: { members: Array<GitlabUser> } = req.body;
        params.members = [ await req.session.profile.gitlabProfile!.value, ...params.members ].removeObjectDuplicates(gitlabUser => gitlabUser.id);
        const assignment: Assignment = req.boundParams.assignment!;


        const reachedLimitUsers: Array<GitlabUser> = await this.checkExerciseLimit(assignment, params.members);
        if ( reachedLimitUsers.length > 0 ) {
            return req.session.sendResponse(res, StatusCodes.INSUFFICIENT_SPACE_ON_RESOURCE, reachedLimitUsers, 'Max exercise per assignment reached', DojoStatusCode.MAX_EXERCISE_PER_ASSIGNMENT_REACHED);
        }


        const exerciseId: string = uuidv4();
        const secret: string = uuidv4();
        const repository: GitlabRepository | undefined = await this.createExerciseRepository(assignment, params.members, exerciseId, req, res);

        if ( !repository ) {
            return;
        }

        await new Promise(resolve => setTimeout(resolve, Config.gitlab.repository.timeoutAfterCreation));

        try {
            await GitlabManager.protectBranch(repository.id, '*', false, GitlabAccessLevel.DEVELOPER, GitlabAccessLevel.DEVELOPER, GitlabAccessLevel.OWNER);

            await GitlabManager.addRepositoryVariable(repository.id, 'DOJO_EXERCISE_ID', exerciseId, false, true);
            await GitlabManager.addRepositoryVariable(repository.id, 'DOJO_SECRET', secret, false, true);
            await GitlabManager.addRepositoryVariable(repository.id, 'DOJO_RESULTS_FOLDER', Config.exercise.pipelineResultsFolder, false, false);

            await GitlabManager.addRepositoryBadge(repository.id, Config.gitlab.badges.pipeline.link, Config.gitlab.badges.pipeline.imageUrl, 'Pipeline Status');
        } catch ( error ) {
            return GlobalHelper.repositoryCreationError('Repo params error', error, req, res, DojoStatusCode.EXERCISE_CREATION_GITLAB_ERROR, DojoStatusCode.EXERCISE_CREATION_INTERNAL_ERROR, repository);
        }

        try {
            await GitlabManager.updateFile(repository.id, '.gitlab-ci.yml', fs.readFileSync(path.join(__dirname, '../../assets/exercise_gitlab_ci.yml'), 'base64'), 'Add .gitlab-ci.yml (DO NOT MODIFY THIS FILE)');
        } catch ( error ) {
            return GlobalHelper.repositoryCreationError('CI file update error', error, req, res, DojoStatusCode.EXERCISE_CREATION_GITLAB_ERROR, DojoStatusCode.EXERCISE_CREATION_INTERNAL_ERROR, repository);
        }

        try {
            await Promise.all([ ...new Set([ ...assignment.staff.map(user => user.id), ...params.members.map(member => member.id) ]) ].map(async (memberId: number): Promise<GitlabMember | false> => {
                try {
                    return await GitlabManager.addRepositoryMember(repository.id, memberId, GitlabAccessLevel.DEVELOPER);
                } catch ( error ) {
                    logger.error('Add member error');
                    logger.error(error);
                    return false;
                }
            }));

            const exercise: Exercise = await db.exercise.create({
                                                                    data: {
                                                                        id                : exerciseId,
                                                                        assignmentName    : assignment.name,
                                                                        name              : repository.name,
                                                                        secret            : secret,
                                                                        gitlabId          : repository.id,
                                                                        gitlabLink        : repository.web_url,
                                                                        gitlabCreationInfo: repository as unknown as Prisma.JsonObject,
                                                                        gitlabLastInfo    : repository as unknown as Prisma.JsonObject,
                                                                        gitlabLastInfoDate: new Date(),
                                                                        members           : {
                                                                            connectOrCreate: [ ...params.members.map(gitlabUser => {
                                                                                return {
                                                                                    create: {
                                                                                        id            : gitlabUser.id,
                                                                                        gitlabUsername: gitlabUser.name
                                                                                    },
                                                                                    where : {
                                                                                        id: gitlabUser.id
                                                                                    }
                                                                                };
                                                                            }) ]
                                                                        }
                                                                    }
                                                                }) as unknown as Exercise;

            return req.session.sendResponse(res, StatusCodes.OK, exercise);
        } catch ( error ) {
            return GlobalHelper.repositoryCreationError('DB error', error, req, res, DojoStatusCode.EXERCISE_CREATION_GITLAB_ERROR, DojoStatusCode.EXERCISE_CREATION_INTERNAL_ERROR, repository);
        }
    }

    private async getAssignment(req: express.Request, res: express.Response) {
        const repoTree: Array<GitlabTreeFile> = await GitlabManager.getRepositoryTree(req.boundParams.exercise!.assignment.gitlabId);

        let assignmentHjsonFile!: GitlabFile;
        const immutableFiles: Array<GitlabFile> = await Promise.all(Config.assignment.baseFiles.map(async (baseFile: string) => {
            const file = await GitlabManager.getFile(req.boundParams.exercise!.assignment.gitlabId, baseFile);

            if ( baseFile === Config.assignment.filename ) {
                assignmentHjsonFile = file;
            }

            return file;
        }));

        const dojoAssignmentFile: AssignmentFile = JSON5.parse(atob(assignmentHjsonFile.content)) as AssignmentFile;

        const immutablePaths = dojoAssignmentFile.immutable.map(fileDescriptor => fileDescriptor.path);

        await Promise.all(repoTree.map(async gitlabTreeFile => {
            if ( gitlabTreeFile.type === GitlabTreeFileType.BLOB ) {
                for ( const immutablePath of immutablePaths ) {
                    if ( gitlabTreeFile.path.startsWith(immutablePath) ) {
                        immutableFiles.push(await GitlabManager.getFile(req.boundParams.exercise!.assignment.gitlabId, gitlabTreeFile.path));
                        break;
                    }
                }
            }
        }));

        return req.session.sendResponse(res, StatusCodes.OK, {
            assignment    : (req.boundParams.exercise as Exercise).assignment,
            assignmentFile: dojoAssignmentFile,
            immutable     : immutableFiles
        });
    }

    private async createResult(req: express.Request, res: express.Response) {
        const params: { exitCode: number, commit: Record<string, string>, results: ExerciseResultsFile, files: Array<IFileDirStat>, archiveBase64: string } = req.body;
        const exercise: Exercise = req.boundParams.exercise!;

        const result = await db.result.create({
                                                  data: {
                                                      exerciseId: exercise.id,
                                                      exitCode  : params.exitCode,
                                                      success   : params.results.success!,
                                                      commit    : params.commit,
                                                      results   : params.results as unknown as Prisma.JsonObject,
                                                      files     : params.files
                                                  }
                                              });

        fs.writeFileSync(path.join(Config.getResultsFolder(exercise), `${ result.dateTime.toISOString().replace(/:/g, '_') }.tar.gz`), params.archiveBase64, 'base64');

        req.session.sendResponse(res, StatusCodes.OK);
    }
}


export default new ExerciseRoutes();
