import { Express }           from 'express-serve-static-core';
import cors                  from 'cors';
import morganMiddleware      from '../logging/MorganMiddleware';
import { AddressInfo }       from 'net';
import http                  from 'http';
import helmet                from 'helmet';
import express               from 'express';
import WorkerTask            from '../process/WorkerTask';
import multer                from 'multer';
import SessionMiddleware     from '../middlewares/SessionMiddleware';
import Config                from '../config/Config';
import logger                from '../shared/logging/WinstonLogger';
import ParamsCallbackManager from '../middlewares/ParamsCallbackManager';
import ApiRoutesManager      from '../routes/ApiRoutesManager';
import compression           from 'compression';


class API implements WorkerTask {
    private readonly backend: Express;
    private server: http.Server | undefined;

    constructor() {
        this.backend = express();

        this.backend.use(multer({
                                    limits: { fieldSize: 100 * 1024 * 1024 }
                                }).none()); //Used for extract params from body with format "form-data", The none is for say that we do not wait a file in params
        this.backend.use(morganMiddleware); //Log API accesses
        this.backend.use(helmet()); //Help to secure express, https://helmetjs.github.io/
        this.backend.use(cors()); //Allow CORS requests
        this.backend.use(compression()); //Compress responses

        ParamsCallbackManager.register(this.backend);

        this.backend.use(SessionMiddleware.register());

        ApiRoutesManager.registerOnBackend(this.backend);
    }

    run() {
        this.server = this.backend.listen(Config.api.port, '0.0.0.0', () => {
            const {
                      port,
                      address
                  } = this.server!.address() as AddressInfo;
            logger.info(`Server started on http://${ address }:${ port }`);
        });
    }
}


export default API;
