import { Express }                    from 'express-serve-static-core';
import express                        from 'express';
import * as ExpressValidator          from 'express-validator';
import { StatusCodes }                from 'http-status-codes';
import RoutesManager                  from '../express/RoutesManager';
import ParamsValidatorMiddleware      from '../middlewares/ParamsValidatorMiddleware';
import SecurityMiddleware             from '../middlewares/SecurityMiddleware';
import SecurityCheckType              from '../types/SecurityCheckType';
import GitlabUser                     from '../shared/types/Gitlab/GitlabUser';
import GitlabManager                  from '../managers/GitlabManager';
import Config                         from '../config/Config';
import GitlabMember                   from '../shared/types/Gitlab/GitlabMember';
import GitlabAccessLevel              from '../shared/types/Gitlab/GitlabAccessLevel';
import GitlabRepository               from '../shared/types/Gitlab/GitlabRepository';
import { AxiosError, HttpStatusCode } from 'axios';
import logger                         from '../shared/logging/WinstonLogger';
import DojoValidators                 from '../helpers/DojoValidators';
import { Prisma }                     from '@prisma/client';
import db                             from '../helpers/DatabaseHelper';
import { Enonce }                     from '../types/DatabaseTypes';
import EnonceManager                  from '../managers/EnonceManager';
import GitlabVisibility               from '../shared/types/Gitlab/GitlabVisibility';


class EnonceRoutes implements RoutesManager {
    private readonly enonceValidator: ExpressValidator.Schema = {
        name    : {
            trim    : true,
            notEmpty: true
        },
        members : {
            trim           : true,
            notEmpty       : true,
            customSanitizer: DojoValidators.jsonSanitizer
        },
        template: {
            trim           : true,
            custom         : DojoValidators.templateUrlValidator,
            customSanitizer: DojoValidators.templateUrlSanitizer
        }
    };

    registerOnBackend(backend: Express) {
        backend.get('/enonces/:enonceNameOrUrl', SecurityMiddleware.check(true), this.getEnonce);
        backend.post('/enonces', SecurityMiddleware.check(true, SecurityCheckType.TEACHING_STAFF), ParamsValidatorMiddleware.validate(this.enonceValidator), this.createEnonce);

        backend.patch('/enonces/:enonceNameOrUrl/publish', SecurityMiddleware.check(true, SecurityCheckType.ENONCE_STAFF), this.changeEnoncePublishedStatus(true));
        backend.patch('/enonces/:enonceNameOrUrl/unpublish', SecurityMiddleware.check(true, SecurityCheckType.ENONCE_STAFF), this.changeEnoncePublishedStatus(false));
    }

    // Get an enonce by its name or gitlab url
    private async getEnonce(req: express.Request, res: express.Response) {
        const enonce: Enonce | undefined = req.boundParams.enonce;

        if ( enonce && !enonce.published && !await EnonceManager.isUserAllowedToAccessEnonce(enonce, req.session.profile) ) {
            // @ts-ignore
            delete enonce.gitlabId;
            // @ts-ignore
            delete enonce.gitlabLink;
            // @ts-ignore
            delete enonce.gitlabCreationInfo;
            // @ts-ignore
            delete enonce.gitlabLastInfo;
            // @ts-ignore
            delete enonce.gitlabLastInfoDate;
            // @ts-ignore
            delete enonce.staff;
            // @ts-ignore
            delete enonce.exercices;
        }

        return enonce ? req.session.sendResponse(res, StatusCodes.OK, enonce) : res.status(StatusCodes.NOT_FOUND).send();
    }

    private async createEnonce(req: express.Request, res: express.Response) {
        const params: {
            name: string, members: Array<GitlabUser>, template: string
        } = req.body;
        params.members = [ await req.session.profile.gitlabProfile.value, ...params.members ];
        params.members = params.members.removeObjectDuplicates(gitlabUser => gitlabUser.id);


        let repository: GitlabRepository;
        try {
            repository = await GitlabManager.createRepository(params.name, Config.enonce.default.description.replace('{{ENONCE_NAME}}', params.name), Config.enonce.default.visibility, Config.enonce.default.initReadme, Config.gitlab.group.enonces, Config.enonce.default.sharedRunnersEnabled, Config.enonce.default.wikiEnabled, params.template);

            await GitlabManager.protectBranch(repository.id, '*', true, GitlabAccessLevel.DEVELOPER, GitlabAccessLevel.DEVELOPER, GitlabAccessLevel.OWNER);
        } catch ( error ) {
            if ( error instanceof AxiosError ) {
                if ( error.response?.data.message.name && error.response.data.message.name == 'has already been taken' ) {
                    return res.status(StatusCodes.CONFLICT).send();
                }

                return res.status(error.response?.status ?? HttpStatusCode.InternalServerError).send();
            }

            logger.error(error);
            return res.status(StatusCodes.INTERNAL_SERVER_ERROR).send();
        }

        try {
            await Promise.all(params.members.map(member => member.id).map(async (memberId: number): Promise<GitlabMember | false> => {
                try {
                    return await GitlabManager.addRepositoryMember(repository.id, memberId, GitlabAccessLevel.DEVELOPER);
                } catch ( e ) {
                    return false;
                }
            }));

            const enonce: Enonce = await db.enonce.create({
                                                              data: {
                                                                  name              : repository.name,
                                                                  gitlabId          : repository.id,
                                                                  gitlabLink        : repository.web_url,
                                                                  gitlabCreationInfo: repository as unknown as Prisma.JsonObject,
                                                                  gitlabLastInfo    : repository as unknown as Prisma.JsonObject,
                                                                  gitlabLastInfoDate: new Date(),
                                                                  staff             : {
                                                                      connectOrCreate: [ ...params.members.map(gitlabUser => {
                                                                          return {
                                                                              create: {
                                                                                  gitlabId : gitlabUser.id,
                                                                                  firstname: gitlabUser.name
                                                                              },
                                                                              where : {
                                                                                  gitlabId: gitlabUser.id
                                                                              }
                                                                          };
                                                                      }) ]
                                                                  }
                                                              }
                                                          }) as unknown as Enonce;

            return req.session.sendResponse(res, StatusCodes.OK, enonce);
        } catch ( error ) {
            if ( error instanceof AxiosError ) {
                return res.status(error.response?.status ?? HttpStatusCode.InternalServerError).send();
            }

            logger.error(error);
            return res.status(StatusCodes.INTERNAL_SERVER_ERROR).send();
        }
    }

    private changeEnoncePublishedStatus(publish: boolean): (req: express.Request, res: express.Response) => Promise<void> {
        return async (req: express.Request, res: express.Response): Promise<void> => {
            try {
                await GitlabManager.changeRepositoryVisibility(req.boundParams.enonce!.gitlabId, publish ? GitlabVisibility.INTERNAL : GitlabVisibility.PRIVATE);

                await db.enonce.update({
                                           where: {
                                               name: req.boundParams.enonce!.name
                                           },
                                           data : {
                                               published: publish
                                           }
                                       });

                req.session.sendResponse(res, StatusCodes.OK);
            } catch ( error ) {
                if ( error instanceof AxiosError ) {
                    res.status(error.response?.status ?? HttpStatusCode.InternalServerError).send();
                    return;
                }

                logger.error(error);
                res.status(StatusCodes.INTERNAL_SERVER_ERROR).send();
            }
        };
    }

}


export default new EnonceRoutes();
