import { Express }                    from 'express-serve-static-core';
import express                        from 'express';
import * as ExpressValidator          from 'express-validator';
import { StatusCodes }                from 'http-status-codes';
import RoutesManager                  from '../express/RoutesManager';
import ParamsValidatorMiddleware      from '../middlewares/ParamsValidatorMiddleware';
import SecurityMiddleware             from '../middlewares/SecurityMiddleware';
import GitlabUser                     from '../shared/types/Gitlab/GitlabUser';
import GitlabManager                  from '../managers/GitlabManager';
import Config                         from '../config/Config';
import GitlabRepository               from '../shared/types/Gitlab/GitlabRepository';
import { AxiosError, HttpStatusCode } from 'axios';
import logger                         from '../shared/logging/WinstonLogger';
import DojoValidators                 from '../helpers/DojoValidators';
import { v4 as uuidv4 }               from 'uuid';
import GitlabMember                   from '../shared/types/Gitlab/GitlabMember';
import GitlabAccessLevel              from '../shared/types/Gitlab/GitlabAccessLevel';
import { Prisma }                     from '@prisma/client';
import { Enonce, Exercice }           from '../types/DatabaseTypes';
import db                             from '../helpers/DatabaseHelper';
import SecurityCheckType              from '../types/SecurityCheckType';
import GitlabTreeFile                 from '../shared/types/Gitlab/GitlabTreeFile';
import GitlabFile                     from '../shared/types/Gitlab/GitlabFile';
import EnonceFile                     from '../shared/types/Dojo/EnonceFile';
import GitlabTreeFileType             from '../shared/types/Gitlab/GitlabTreeFileType';
import JSON5                          from 'json5';
import ExerciceResultsFile            from '../shared/types/Dojo/ExerciceResultsFile';
import fs                             from 'fs';
import path                           from 'path';


class ExerciceRoutes implements RoutesManager {
    private readonly exerciceValidator: ExpressValidator.Schema = {
        members: {
            trim           : true,
            notEmpty       : true,
            customSanitizer: DojoValidators.jsonSanitizer
        }
    };

    private readonly resultValidator: ExpressValidator.Schema = {
        exitCode     : {
            isInt: true,
            toInt: true
        },
        commit       : {
            trim           : true,
            notEmpty       : true,
            customSanitizer: DojoValidators.jsonSanitizer
        },
        results      : {
            trim           : true,
            notEmpty       : true,
            custom         : DojoValidators.exerciceResultsValidator,
            customSanitizer: DojoValidators.jsonSanitizer
        },
        files        : {
            trim           : true,
            notEmpty       : true,
            customSanitizer: DojoValidators.jsonSanitizer
        },
        archiveBase64: {
            isBase64: true,
            notEmpty: true
        }
    };

    registerOnBackend(backend: Express) {
        backend.post('/enonces/:enonceNameOrUrl/exercices', SecurityMiddleware.check(true, SecurityCheckType.ENONCE_IS_PUBLISHED), ParamsValidatorMiddleware.validate(this.exerciceValidator), this.createExercice.bind(this));

        backend.get('/exercices/:exerciceId/enonce', SecurityMiddleware.check(false, SecurityCheckType.EXERCICE_SECRET), this.getEnonce.bind(this));

        backend.post('/exercices/:exerciceId/results', SecurityMiddleware.check(false, SecurityCheckType.EXERCICE_SECRET), ParamsValidatorMiddleware.validate(this.resultValidator), this.createResult.bind(this));
    }

    private getExerciceName(enonce: Enonce, members: Array<GitlabUser>, suffix: number): string {
        return `DojoEx - ${ enonce.name } - ${ members.map(member => member.username).join(' + ') }${ suffix > 0 ? ` - ${ suffix }` : '' }`;
    }

    private getExercicePath(enonce: Enonce, exerciceId: string): string {
        return `dojo-ex_${ (enonce.gitlabLastInfo as unknown as GitlabRepository).path }_${ exerciceId }`;
    }

    private async createExercice(req: express.Request, res: express.Response) {
        const params: { members: Array<GitlabUser> } = req.body;
        params.members = [ await req.session.profile.gitlabProfile!.value, ...params.members ].removeObjectDuplicates(gitlabUser => gitlabUser.id);
        const enonce: Enonce = req.boundParams.enonce!;


        const exerciceId: string = uuidv4();
        const secret: string = uuidv4();
        let repository!: GitlabRepository;

        let suffix: number = 0;
        do {
            try {
                repository = await GitlabManager.forkRepository((enonce.gitlabCreationInfo as unknown as GitlabRepository).id, this.getExerciceName(enonce, params.members, suffix), this.getExercicePath(req.boundParams.enonce!, exerciceId), Config.exercice.default.description.replace('{{ENONCE_NAME}}', enonce.name), Config.exercice.default.visibility, Config.gitlab.group.exercices);

                await GitlabManager.protectBranch(repository.id, '*', false, GitlabAccessLevel.DEVELOPER, GitlabAccessLevel.DEVELOPER, GitlabAccessLevel.OWNER);

                await GitlabManager.addRepositoryVariable(repository.id, 'DOJO_EXERCICE_ID', exerciceId, false, true);
                await GitlabManager.addRepositoryVariable(repository.id, 'DOJO_SECRET', secret, false, true);
                await GitlabManager.addRepositoryVariable(repository.id, 'DOJO_RESULTS_FOLDER', Config.exercice.pipelineResultsFolder, false, false);

                break;
            } catch ( error ) {
                if ( error instanceof AxiosError ) {
                    if ( error.response?.data.message.name && error.response.data.message.name == 'has already been taken' ) {
                        suffix++;
                    } else {
                        return res.status(error.response?.status ?? HttpStatusCode.InternalServerError).send();
                    }
                } else {
                    return res.status(StatusCodes.INTERNAL_SERVER_ERROR).send();
                }
            }
        } while ( suffix < Config.exercice.maxSameName );

        if ( suffix >= Config.exercice.maxSameName ) {
            return res.status(StatusCodes.INSUFFICIENT_SPACE_ON_RESOURCE).send();
        }

        try {
            await GitlabManager.createFile(repository.id, '.gitlab-ci.yml', fs.readFileSync(path.join(__dirname, '../../assets/exercice_gitlab_ci.yml'), 'base64'), 'Add .gitlab-ci.yml (DO NOT MODIFY THIS FILE)');
        } catch ( error ) {
            logger.error(error);

            if ( error instanceof AxiosError ) {
                return res.status(error.response?.status ?? HttpStatusCode.InternalServerError).send();
            }

            return res.status(StatusCodes.INTERNAL_SERVER_ERROR).send();
        }

        try {
            await Promise.all([ ...new Set([ ...enonce.staff.map(user => user.gitlabId), ...params.members.map(member => member.id) ]) ].map(async (memberId: number): Promise<GitlabMember | false> => {
                try {
                    return await GitlabManager.addRepositoryMember(repository.id, memberId, GitlabAccessLevel.DEVELOPER);
                } catch ( e ) {
                    return false;
                }
            }));

            const exercice: Exercice = await db.exercice.create({
                                                                    data: {
                                                                        id                : exerciceId,
                                                                        enonceName        : enonce.name,
                                                                        name              : repository.name,
                                                                        secret            : secret,
                                                                        gitlabId          : repository.id,
                                                                        gitlabLink        : repository.web_url,
                                                                        gitlabCreationInfo: repository as unknown as Prisma.JsonObject,
                                                                        gitlabLastInfo    : repository as unknown as Prisma.JsonObject,
                                                                        gitlabLastInfoDate: new Date(),
                                                                        members           : {
                                                                            connectOrCreate: [ ...params.members.map(gitlabUser => {
                                                                                return {
                                                                                    create: {
                                                                                        gitlabId : gitlabUser.id,
                                                                                        firstname: gitlabUser.name
                                                                                    },
                                                                                    where : {
                                                                                        gitlabId: gitlabUser.id
                                                                                    }
                                                                                };
                                                                            }) ]
                                                                        }
                                                                    }
                                                                }) as unknown as Exercice;

            return req.session.sendResponse(res, StatusCodes.OK, exercice);
        } catch ( error ) {
            logger.error(error);
            
            if ( error instanceof AxiosError ) {
                return res.status(error.response?.status ?? HttpStatusCode.InternalServerError).send();
            }

            return res.status(StatusCodes.INTERNAL_SERVER_ERROR).send();
        }
    }

    private async getEnonce(req: express.Request, res: express.Response) {
        const repoTree: Array<GitlabTreeFile> = await GitlabManager.getRepositoryTree(req.boundParams.exercice!.enonce.gitlabId);

        let enonceHjsonFile!: GitlabFile;
        let immutableFiles: Array<GitlabFile> = await Promise.all(Config.enonce.baseFiles.map(async (baseFile: string) => {
            let file = await GitlabManager.getFile(req.boundParams.exercice!.enonce.gitlabId, baseFile);

            if ( baseFile === Config.enonce.filename ) {
                enonceHjsonFile = file;
            }

            return file;
        }));

        const dojoEnonceFile: EnonceFile = JSON5.parse(atob(enonceHjsonFile.content)) as EnonceFile;

        const immutablePaths = dojoEnonceFile.immutable.map(fileDescriptor => fileDescriptor.path);

        await Promise.all(repoTree.map(async gitlabTreeFile => {
            if ( gitlabTreeFile.type == GitlabTreeFileType.BLOB ) {
                for ( const immutablePath of immutablePaths ) {
                    if ( gitlabTreeFile.path.startsWith(immutablePath) ) {
                        immutableFiles.push(await GitlabManager.getFile(req.boundParams.exercice!.enonce.gitlabId, gitlabTreeFile.path));
                        break;
                    }
                }
            }
        }));

        return req.session.sendResponse(res, StatusCodes.OK, {
            enonce    : (req.boundParams.exercice as Exercice).enonce,
            enonceFile: dojoEnonceFile,
            immutable : immutableFiles
        });
    }

    private async createResult(req: express.Request, res: express.Response) {
        const params: { exitCode: number, commit: any, results: ExerciceResultsFile, files: any, archiveBase64: string } = req.body;
        const exercice: Exercice = req.boundParams.exercice!;

        const result = await db.result.create({
                                                  data: {
                                                      exerciceId: exercice.id,
                                                      exitCode  : params.exitCode,
                                                      success   : params.results.success,
                                                      commit    : params.commit,
                                                      results   : params.results as unknown as Prisma.JsonObject,
                                                      files     : params.files
                                                  }
                                              });

        fs.writeFileSync(path.join(Config.getResultsFolder(exercice), `${ result.dateTime.toISOString().replace(/:/g, '_') }.tar.gz`), params.archiveBase64, 'base64');

        req.session.sendResponse(res, StatusCodes.OK);
    }
}


export default new ExerciceRoutes();
