import { Express }               from 'express-serve-static-core';
import express                   from 'express';
import * as ExpressValidator     from 'express-validator';
import { StatusCodes }           from 'http-status-codes';
import * as bcrypt               from 'bcryptjs';
import RoutesManager             from '../express/RoutesManager';
import ParamsValidatorMiddleware from '../middlewares/ParamsValidatorMiddleware';
import UserManager               from '../managers/UserManager';
import SecurityMiddleware        from '../middlewares/SecurityMiddleware';
import { User }                  from '../types/DatabaseTypes';


class SessionRoutes implements RoutesManager {
    private readonly loginValidator: ExpressValidator.Schema = {
        user    : {
            trim    : true,
            notEmpty: true
        },
        password: {
            trim    : true,
            notEmpty: true
        }
    };

    registerOnBackend(backend: Express) {
        backend.post('/login', ParamsValidatorMiddleware.validate(this.loginValidator), this.login);
        backend.get('/test_session', SecurityMiddleware.check(true), (req: express.Request, res: express.Response) => req.session.sendResponse(res, StatusCodes.OK));
    }

    private async login(req: express.Request, res: express.Response) {
        const params: {
            user: string, password: string
        } = req.body;

        const user: User | undefined = await UserManager.getByMail(params.user);

        if ( user ) {
            if ( bcrypt.compareSync(params.password, user.password ?? '') ) {
                req.session.profile = user;

                req.session.sendResponse(res, StatusCodes.OK);
                return;
            }
        }

        req.session.sendResponse(res, StatusCodes.NOT_FOUND);
    }
}


export default new SessionRoutes();
