import { Express }               from 'express-serve-static-core';
import express                   from 'express';
import * as ExpressValidator     from 'express-validator';
import { StatusCodes }           from 'http-status-codes';
import RoutesManager             from '../express/RoutesManager';
import ParamsValidatorMiddleware from '../middlewares/ParamsValidatorMiddleware';
import SecurityMiddleware        from '../middlewares/SecurityMiddleware';
import GitlabManager             from '../managers/GitlabManager';
import UserManager               from '../managers/UserManager';
import DojoStatusCode            from '../shared/types/Dojo/DojoStatusCode';
import Config                    from '../config/Config';
import GlobalHelper              from '../helpers/GlobalHelper';


class SessionRoutes implements RoutesManager {
    private readonly loginValidator: ExpressValidator.Schema = {
        accessToken : {
            trim    : true,
            notEmpty: true
        },
        refreshToken: {
            trim    : true,
            notEmpty: true
        }
    };

    private readonly refreshTokensValidator: ExpressValidator.Schema = {
        refreshToken: {
            trim    : true,
            notEmpty: true
        }
    };

    registerOnBackend(backend: Express) {
        backend.post('/login', ParamsValidatorMiddleware.validate(this.loginValidator), this.login.bind(this));
        backend.post('/refresh_tokens', ParamsValidatorMiddleware.validate(this.refreshTokensValidator), this.refreshTokens.bind(this));
        backend.get('/test_session', SecurityMiddleware.check(true), this.testSession.bind(this));
    }

    private async login(req: express.Request, res: express.Response) {
        try {
            const params: {
                accessToken: string, refreshToken: string
            } = req.body;

            const gitlabUser = await GitlabManager.getUserProfile(params.accessToken);

            if ( gitlabUser ) {
                req.session.profile = await UserManager.getUpdateFromGitlabProfile(gitlabUser);

                req.session.sendResponse(res, StatusCodes.OK);
                return;
            } else {
                req.session.sendResponse(res, StatusCodes.NOT_FOUND);
            }
        } catch ( error ) {
            req.session.sendResponse(res, StatusCodes.INTERNAL_SERVER_ERROR, {}, 'Unknown error while logging in', DojoStatusCode.LOGIN_FAILED);
        }
    }

    private async refreshTokens(req: express.Request, res: express.Response) {
        try {
            const params: {
                refreshToken: string
            } = req.body;

            const gitlabTokens = await GlobalHelper.sharedGitlabManager.getTokens(params.refreshToken, true, Config.login.gitlab.client.secret);

            req.session.sendResponse(res, StatusCodes.OK, gitlabTokens);
        } catch ( error ) {
            req.session.sendResponse(res, StatusCodes.INTERNAL_SERVER_ERROR, {}, 'Unknown error while refresh tokens', DojoStatusCode.REFRESH_TOKENS_FAILED);
        }
    }

    private async testSession(req: express.Request, res: express.Response) {
        req.session.sendResponse(res, StatusCodes.OK);
    }
}


export default new SessionRoutes();
