import { getReasonPhrase, StatusCodes } from 'http-status-codes';
import * as jwt                         from 'jsonwebtoken';
import { JwtPayload }                   from 'jsonwebtoken';
import Config                           from '../config/Config.js';
import express                          from 'express';
import UserManager                      from '../managers/UserManager.js';
import { User }                         from '../types/DatabaseTypes.js';
import DojoBackendResponse              from '../shared/types/Dojo/DojoBackendResponse.js';


class Session {
    private _profile!: User;

    get profile(): User {
        return this._profile;
    }

    set profile(newProfile: User) {
        this._profile = newProfile;
    }

    async initSession(req: express.Request, res: express.Response) {
        const authorization = req.headers.authorization;
        if ( authorization && authorization.startsWith('Bearer ') ) {
            const jwtToken = authorization.replace('Bearer ', '');

            try {
                const jwtData = jwt.verify(jwtToken, Config.jwtConfig.secret) as JwtPayload;

                if ( jwtData.profile ) {
                    this.profile = jwtData.profile;
                    this.profile = await UserManager.getById(this.profile.id) ?? this.profile;
                }
            } catch ( err ) {
                res.sendStatus(StatusCodes.UNAUTHORIZED).end();
            }
        }
    }

    private static getToken(profileJson: unknown): string | null {
        const options = Config.jwtConfig.expiresIn > 0 ? { expiresIn: Config.jwtConfig.expiresIn } : {};
        return profileJson === null ? null : jwt.sign({ profile: profileJson }, Config.jwtConfig.secret, options);
    }

    private async getResponse<T>(code: number, data: T, descriptionOverride?: string): Promise<DojoBackendResponse<T>> {
        const profileJson = this.profile;

        let reasonPhrase = '';

        try {
            reasonPhrase = getReasonPhrase(code);
        } catch { /* empty */ }

        return {
            timestamp   : (new Date()).toISOString(),
            code        : code,
            description : descriptionOverride ? descriptionOverride : reasonPhrase,
            sessionToken: Session.getToken(profileJson),
            data        : data
        };
    }

    /*
     Send a response to the client
     Information: Data could be a promise or an object. If it's a promise, we wait on the data to be resolved before sending the response
     */
    sendResponse(res: express.Response | undefined, code: number, data?: unknown, descriptionOverride?: string, internalCode?: number) {
        if ( res ) {
            Promise.resolve(data).then((toReturn: unknown) => {
                this.getResponse(internalCode ?? code, toReturn, descriptionOverride).then(response => {
                    res.status(code).json(response);
                });
            });
        }
    }
}


export default Session;
