import express         from 'express';
import Config          from '../config/Config.js';
import semver          from 'semver/preload';
import Session         from '../controllers/Session.js';
import DojoStatusCode  from '../shared/types/Dojo/DojoStatusCode.js';
import { StatusCodes } from 'http-status-codes';


class ClientVersionCheckerMiddleware {
    register(): (req: express.Request, res: express.Response, next: express.NextFunction) => void {
        return (req: express.Request, res: express.Response, next: express.NextFunction) => {
            if ( req.headers['client'] && req.headers['client-version'] ) {
                const requestClient = req.headers['client'] as string;
                const requestClientVersion = req.headers['client-version'] as string;

                for ( const [ client, condition ] of Object.entries(Config.requestClientValidation.version) ) {
                    if ( requestClient === client ) {
                        if ( semver.satisfies(semver.valid(semver.coerce(requestClientVersion)) || '0', condition) ) {
                            next();
                            return;
                        } else {
                            new Session().sendResponse(res, StatusCodes.METHOD_NOT_ALLOWED, {}, `Client version ${ requestClientVersion } is not supported. Please update your client.`, DojoStatusCode.CLIENT_VERSION_NOT_SUPPORTED);
                            return;
                        }
                    }
                }

                new Session().sendResponse(res, StatusCodes.METHOD_NOT_ALLOWED, {}, `Unsupported client.`, DojoStatusCode.CLIENT_NOT_SUPPORTED);
            } else {
                next();
            }
        };
    }
}


export default new ClientVersionCheckerMiddleware();