import { Prisma }           from '@prisma/client';
import { Assignment, User } from '../types/DatabaseTypes.js';
import db                   from '../helpers/DatabaseHelper.js';


class AssignmentManager {
    async isUserAllowedToAccessAssignment(assignment: Assignment, user: User): Promise<boolean> {
        if ( !assignment.staff ) {
            assignment.staff = await db.assignment.findUnique({
                                                                  where: {
                                                                      name: assignment.name
                                                                  }
                                                              }).staff() ?? [];
        }
        return assignment.staff.findIndex(staff => staff.id === user.id) !== -1;
    }

    async getByName(name: string, include: Prisma.AssignmentInclude | undefined = undefined): Promise<Assignment | undefined> {
        return await db.assignment.findUnique({
                                                  where  : {
                                                      name: name
                                                  },
                                                  include: include
                                              }) as unknown as Assignment ?? undefined;
    }

    async getByGitlabLink(gitlabLink: string, include: Prisma.AssignmentInclude | undefined = undefined): Promise<Assignment | undefined> {
        const nameInUrl = gitlabLink.replace('.git', '').split('/').pop()!;

        const result = await db.assignment.findMany({
                                                        where  : {
                                                            gitlabLink: {
                                                                endsWith: `/${ nameInUrl }`
                                                            }
                                                        },
                                                        include: include
                                                    }) as Array<Assignment>;

        return result.length > 0 ? result[0] : undefined;
    }

    get(nameOrUrl: string, include: Prisma.AssignmentInclude | undefined = undefined): Promise<Assignment | undefined> {
        if ( nameOrUrl.includes('://') ) {
            return this.getByGitlabLink(nameOrUrl, include);
        } else {
            return this.getByName(nameOrUrl, include);
        }
    }
}


export default new AssignmentManager();
