import GitlabUser from '../shared/types/Gitlab/GitlabUser';
import { Prisma } from '@prisma/client';
import db         from '../helpers/DatabaseHelper';
import { User }   from '../types/DatabaseTypes';


class UserManager {
    async getByMail(mail: string, include: Prisma.UserInclude | undefined = undefined): Promise<User | undefined> {
        return db.user.findUnique({
                                      where  : {
                                          mail: mail
                                      },
                                      include: include
                                  });
    }

    async getById(id: number, include: Prisma.UserInclude | undefined = undefined): Promise<User | undefined> {
        return await db.user.findUnique({
                                            where  : {
                                                id: id
                                            },
                                            include: include
                                        });
    }

    async getByGitlabId(gitlabId: number, returnIdIfUndefined: boolean = true, include: Prisma.UserInclude | undefined = undefined): Promise<User | number | undefined> {
        return (await db.user.findUnique({
                                             where  : {
                                                 gitlabId: gitlabId
                                             },
                                             include: include
                                         })) ?? (returnIdIfUndefined ? gitlabId : undefined);
    }

    async getFromGitlabUser(gitlabUser: GitlabUser, createIfNotExist: boolean = false, include: Prisma.UserInclude | undefined = undefined): Promise<User | number> {
        let user = await this.getByGitlabId(gitlabUser.id, true, include);

        if ( typeof user === 'number' && createIfNotExist ) {
            user = (await db.user.create({
                                             data: {
                                                 firstname: gitlabUser.name,
                                                 gitlabId : gitlabUser.id
                                             }
                                         })).id;
        }

        return user;
    }

    async getFromGitlabUsers(gitlabUsers: Array<GitlabUser>, createIfNotExist: boolean = false, include: Prisma.UserInclude | undefined = undefined): Promise<Array<User | number>> {
        return Promise.all(gitlabUsers.map(gitlabUser => this.getFromGitlabUser(gitlabUser, createIfNotExist, include)));
    }
}


export default new UserManager();
