import express, { RequestHandler } from 'express';
import { TagType }                 from '@prisma/client';
import * as ExpressValidator       from 'express-validator';
import { StatusCodes }             from 'http-status-codes';
import RoutesManager               from '../express/RoutesManager';
import { Express }                 from 'express-serve-static-core';
import db                          from '../helpers/DatabaseHelper';
import SecurityCheckType           from '../types/SecurityCheckType';
import SecurityMiddleware          from '../middlewares/SecurityMiddleware';
import ParamsValidatorMiddleware   from '../middlewares/ParamsValidatorMiddleware';
import DojoStatusCode              from '../shared/types/Dojo/DojoStatusCode';


class TagRoutes implements RoutesManager {
    private readonly tagValidator: ExpressValidator.Schema = {
        name: {
            trim    : true,
            notEmpty: true
        },
        type: {
            trim    : true,
            notEmpty: true
        }
    };

    private readonly tagProposalAnswerValidator: ExpressValidator.Schema = {
        state  : {
            trim    : true,
            notEmpty: true
        },
        details: {
            trim    : true,
            optional: true
        }
    };

    registerOnBackend(backend: Express) {
        backend.post('/tags', SecurityMiddleware.check(true, SecurityCheckType.TEACHING_STAFF), ParamsValidatorMiddleware.validate(this.tagValidator), this.createTag.bind(this) as RequestHandler);
        backend.delete('/tags/:tagName', SecurityMiddleware.check(true, SecurityCheckType.ADMIN), this.deleteTag.bind(this) as RequestHandler);
        backend.get('/tags/proposals', SecurityMiddleware.check(true, SecurityCheckType.ADMIN), this.getTagProposals.bind(this) as RequestHandler);
        backend.post('/tags/proposals', SecurityMiddleware.check(true, SecurityCheckType.TEACHING_STAFF), ParamsValidatorMiddleware.validate(this.tagValidator), this.createTagProposal.bind(this) as RequestHandler);
        backend.patch('/tags/proposals/:tagProposalName', SecurityMiddleware.check(true, SecurityCheckType.ADMIN), ParamsValidatorMiddleware.validate(this.tagProposalAnswerValidator), this.validateTag.bind(this) as RequestHandler);
    }

    private async createTag(req: express.Request, res: express.Response) {
        const tagName = req.body.name;
        const tagType = (req.body.type as string).toUpperCase() as TagType;

        if ( tagType !== TagType.USERDEFINED && !req.session.profile.isAdmin ) {
            return req.session.sendResponse(res, StatusCodes.FORBIDDEN, {}, 'Only admins can create non userDefined tags', DojoStatusCode.TAG_ONLY_ADMIN_CREATION);
        }

        await db.tag.create({
                                data: {
                                    name: tagName,
                                    type: tagType
                                }
                            });

        return req.session.sendResponse(res, StatusCodes.OK);
    }

    private async deleteTag(req: express.Request, res: express.Response) {
        if ( req.boundParams.tag!.assignments.length > 0 ) {
            return req.session.sendResponse(res, StatusCodes.LOCKED, {}, 'Tag is used in assignments', DojoStatusCode.TAG_WITH_ACTIVE_LINK_DELETION);
        }

        await db.tag.delete({
                                where: { name: req.boundParams.tag!.name }
                            });

        return req.session.sendResponse(res, StatusCodes.OK);
    }

    private async getTagProposals(req: express.Request, res: express.Response) {
        const state = req.query.stateFilter as string;

        const tagProposals = await db.tagProposal.findMany(state ? {
            where: { state: state }
        } : {});

        return req.session.sendResponse(res, StatusCodes.OK, tagProposals);
    }

    private async createTagProposal(req: express.Request, res: express.Response) {
        const tagName = req.body.name;
        const tagType = (req.body.type as string).toUpperCase() as TagType;

        await db.tagProposal.create({
                                        data: {
                                            name: tagName,
                                            type: tagType
                                        }
                                    });

        return req.session.sendResponse(res, StatusCodes.OK);
    }

    private async validateTag(req: express.Request, res: express.Response) {
        if ( req.boundParams.tagProposal!.state === 'PendingApproval' ) {
            const state: string = req.body.state;

            if ( state === 'Approved' ) {
                try {
                    await db.tag.create({
                                            data: {
                                                name: req.boundParams.tagProposal!.name,
                                                type: req.boundParams.tagProposal!.type
                                            }
                                        });
                } catch ( error ) {
                    // empty
                }
            }

            await db.tagProposal.update({
                                            where: { name: req.boundParams.tagProposal?.name },
                                            data : {
                                                state  : state,
                                                details: req.body.details ?? ''
                                            }
                                        });

            return req.session.sendResponse(res, StatusCodes.OK);
        } else {
            return req.session.sendResponse(res, StatusCodes.BAD_REQUEST, {}, 'Tag proposal is not pending', DojoStatusCode.TAG_PROPOSAL_ANSWER_NOT_PENDING);
        }
    }
}


export default new TagRoutes();
