import { Express }                 from 'express-serve-static-core';
import express, { RequestHandler } from 'express';
import { StatusCodes }             from 'http-status-codes';
import RoutesManager               from '../express/RoutesManager.js';
import SecurityMiddleware          from '../middlewares/SecurityMiddleware';
import * as ExpressValidator       from 'express-validator';
import ParamsValidatorMiddleware   from '../middlewares/ParamsValidatorMiddleware';
import { Prisma, UserRole }        from '@prisma/client';
import UserManager                 from '../managers/UserManager';


class UserRoutes implements RoutesManager {
    private readonly usersGetValidator: ExpressValidator.Schema = {
        role: {
            trim    : true,
            notEmpty: false,
            optional: true
        }
    };

    registerOnBackend(backend: Express) {
        backend.get('/users', SecurityMiddleware.check(true), ParamsValidatorMiddleware.validate(this.usersGetValidator), this.getUsers.bind(this) as RequestHandler);
    }

    private async getUsers(req: express.Request, res: express.Response) {

        let roleFilter: Prisma.UserWhereInput | undefined = undefined;

        if ( req.query.role ) {
            if ( req.query.role === UserRole.ADMIN ) {
                roleFilter = {
                    role: UserRole.ADMIN
                };
            } else if ( req.query.role === UserRole.TEACHING_STAFF ) {
                roleFilter = {
                    OR: [ {
                        role: UserRole.ADMIN
                    }, {
                        role: UserRole.TEACHING_STAFF
                    } ]
                };
            } else if ( req.query.role === UserRole.STUDENT ) {
                roleFilter = {
                    role: UserRole.STUDENT
                };
            } else {
                return req.session.sendResponse(res, StatusCodes.FORBIDDEN);
            }
        } else if ( !req.session.profile.isAdmin ) {
            return req.session.sendResponse(res, StatusCodes.FORBIDDEN);
        }

        return req.session.sendResponse(res, StatusCodes.OK, await UserManager.getFiltered(roleFilter));
    }
}


export default new UserRoutes();
