import { Express }                    from 'express-serve-static-core';
import cors                           from 'cors';
import morganMiddleware               from '../logging/MorganMiddleware';
import { AddressInfo }                from 'net';
import http                           from 'http';
import helmet                         from 'helmet';
import express                        from 'express';
import WorkerTask                     from '../process/WorkerTask';
import multer                         from 'multer';
import SessionMiddleware              from '../middlewares/SessionMiddleware';
import Config                         from '../config/Config';
import logger                         from '../shared/logging/WinstonLogger';
import ParamsCallbackManager          from '../middlewares/ParamsCallbackManager';
import ApiRoutesManager               from '../routes/ApiRoutesManager';
import compression                    from 'compression';
import ClientVersionCheckerMiddleware from '../middlewares/ClientVersionCheckerMiddleware';
import swaggerUi                      from 'swagger-ui-express';
import path                           from 'path';
import DojoCliVersionHelper           from '../helpers/DojoCliVersionHelper';


class API implements WorkerTask {
    private readonly backend: Express;
    private server: http.Server | undefined;

    constructor() {
        this.backend = express();

        this.initOpenAPI();
        this.initBaseMiddlewares();

        this.backend.use(ClientVersionCheckerMiddleware.register());

        ParamsCallbackManager.registerOnBackend(this.backend);
        SessionMiddleware.registerOnBackend(this.backend);
        ApiRoutesManager.registerOnBackend(this.backend);
    }

    private initBaseMiddlewares() {
        this.backend.use(multer({
                                    limits: {
                                        fieldSize: 15728640 // 15MB
                                    }
                                }).none()); //Used for extract params from body with format "form-data", The none is for say that we do not wait a file in params
        this.backend.use(morganMiddleware); //Log API accesses
        this.backend.use(helmet()); //Help to secure express, https://helmetjs.github.io/
        this.backend.use(cors()); //Allow CORS requests
        this.backend.use(compression()); //Compress responses

        this.backend.use(async (_req, res, next) => {
            res.header('dojocli-latest-version', await DojoCliVersionHelper.getLatestVersion());
            next();
        });
    }

    private initOpenAPI() {
        const options = {
            customSiteTitle: 'Dojo API',
            explorer       : false,
            swaggerOptions : {
                url: '../OpenAPI.yaml'
            }
        };
        this.backend.get('/docs/OpenAPI.yaml', (_req, res) => res.sendFile(path.resolve(__dirname + '/../../assets/OpenAPI/OpenAPI.yaml')));
        this.backend.use('/docs/swagger', swaggerUi.serveFiles(undefined, options), swaggerUi.setup(undefined, options));
        this.backend.get('/docs/redoc.html', (_req, res) => res.sendFile(path.resolve(__dirname + '/../../assets/OpenAPI/redoc.html')));

        this.backend.get('/docs/', (req, res) => {
            const prefix = req.url.slice(-1) === '/' ? '' : 'docs/';
            res.send(`
                <!DOCTYPE html>
                <html lang="en">
                <body>
                    <ul>
                        <li><a href="${ prefix }OpenAPI.yaml">OpenAPI</a></li>
                        <li>GUI
                            <ul>
                                <li><a href="${ prefix }swagger/">Swagger</a></li>
                                <li><a href="${ prefix }redoc.html">Redoc</a></li>
                            </ul>
                        </li>
                    </ul>
                </body>
                </html>
            `);
        });
    }

    run() {
        this.server = this.backend.listen(Config.api.port, '0.0.0.0', () => {
            const {
                      port,
                      address
                  } = this.server!.address() as AddressInfo;
            logger.info(`Server started on http://${ address }:${ port }`);
        });
    }
}


export default API;
