import Config                                                                                                                                                                                 from '../config/Config';
import { StatusCodes }                                                                                                                                                                        from 'http-status-codes';
import GitlabVisibility                                                                                                                                                                       from '../shared/types/Gitlab/GitlabVisibility';
import express                                                                                                                                                                                from 'express';
import SharedConfig                                                                                                                                                                           from '../shared/config/SharedConfig';
import { CommitSchema, ExpandedUserSchema, Gitlab, MemberSchema, ProjectBadgeSchema, ProjectSchema, ReleaseSchema, RepositoryFileExpandedSchema, RepositoryFileSchema, RepositoryTreeSchema } from '@gitbeaker/rest';
import logger                                                                                                                                                                                 from '../shared/logging/WinstonLogger';
import { AccessLevel, EditProjectOptions, ProjectVariableSchema, ProtectedBranchAccessLevel, ProtectedBranchSchema }                                                                          from '@gitbeaker/core';
import GlobalHelper                                                                                                                                                                           from '../helpers/GlobalHelper';


class GitlabManager {
    private readonly api = new Gitlab({
                                          host : SharedConfig.gitlab.URL,
                                          token: Config.gitlab.account.token
                                      });

    public async getUserProfile(token: string): Promise<ExpandedUserSchema | undefined> {
        try {
            const profileApi = new Gitlab({
                                              host : SharedConfig.gitlab.URL,
                                              token: token
                                          });

            return await profileApi.Users.showCurrentUser();
        } catch ( e ) {
            return undefined;
        }
    }

    async getRepositoryMembers(idOrNamespace: string): Promise<Array<MemberSchema>> {
        return await this.api.ProjectMembers.all(idOrNamespace, { includeInherited: true });
    }

    async getRepositoryReleases(repoId: number): Promise<Array<ReleaseSchema>> {
        return await this.api.ProjectReleases.all(repoId);
    }

    async getRepositoryLastCommit(repoId: number, branch: string = 'main'): Promise<CommitSchema | undefined> {
        try {
            const commits = await this.api.Commits.all(repoId, {
                refName : branch,
                maxPages: 1,
                perPage : 1
            });

            return commits.length > 0 ? commits[0] : undefined;
        } catch ( e ) {
            logger.error(e);
            return undefined;
        }
    }

    async createRepository(name: string, description: string, visibility: 'public' | 'internal' | 'private', initializeWithReadme: boolean, namespace: number, sharedRunnersEnabled: boolean, wikiEnabled: boolean, import_url: string): Promise<ProjectSchema> {
        return await this.api.Projects.create({
                                                  name                : name,
                                                  description         : description,
                                                  importUrl           : import_url,
                                                  initializeWithReadme: initializeWithReadme,
                                                  namespaceId         : namespace,
                                                  sharedRunnersEnabled: sharedRunnersEnabled,
                                                  visibility          : visibility,
                                                  wikiAccessLevel     : wikiEnabled ? 'enabled' : 'disabled'
                                              });
    }

    async deleteRepository(repoId: number): Promise<void> {
        return await this.api.Projects.remove(repoId);
    }

    async forkRepository(forkId: number, name: string, path: string, description: string, visibility: 'public' | 'internal' | 'private', namespace: number): Promise<ProjectSchema> {
        return await this.api.Projects.fork(forkId, {
            name       : name,
            path       : path,
            description: description,
            namespaceId: namespace,
            visibility : visibility
        });
    }

    async editRepository(repoId: number, newAttributes: EditProjectOptions): Promise<ProjectSchema> {
        return await this.api.Projects.edit(repoId, newAttributes);
    }

    async changeRepositoryVisibility(repoId: number, visibility: GitlabVisibility): Promise<ProjectSchema> {
        return await this.editRepository(repoId, { visibility: visibility });
    }

    async addRepositoryMember(repoId: number, userId: number, accessLevel: Exclude<AccessLevel, AccessLevel.ADMIN>): Promise<MemberSchema> {
        return await this.api.ProjectMembers.add(repoId, userId, accessLevel);
    }

    async addRepositoryVariable(repoId: number, key: string, value: string, isProtected: boolean, isMasked: boolean): Promise<ProjectVariableSchema> {
        return await this.api.ProjectVariables.create(repoId, key, value, {
            variableType: 'env_var',
            protected   : isProtected,
            masked      : isMasked
        });
    }

    async addRepositoryBadge(repoId: number, linkUrl: string, imageUrl: string, name: string): Promise<ProjectBadgeSchema> {
        return await this.api.ProjectBadges.add(repoId, linkUrl, imageUrl, {
            name: name
        });
    }

    async checkTemplateAccess(projectIdOrNamespace: string, req: express.Request, res?: express.Response): Promise<boolean> {
        // Get the Gitlab project and check if it have public or internal visibility
        try {
            const project: ProjectSchema = await GlobalHelper.sharedGitlabManager.getRepository(projectIdOrNamespace);

            if ( [ 'public', 'internal' ].includes(project.visibility) ) {
                return StatusCodes.OK;
            }
        } catch ( e ) {
            req.session.sendResponse(res, StatusCodes.NOT_FOUND, undefined, 'Template not found', DojoStatusCode.GITLAB_TEMPLATE_NOT_FOUND);
            return false;
        }

        // Check if the user and dojo are members (with at least reporter access) of the project
        const members = await this.getRepositoryMembers(projectIdOrNamespace);
        const isUsersAtLeastReporter = {
            user: false,
            dojo: false
        };
        members.forEach(member => {
            if ( member.access_level >= AccessLevel.REPORTER ) {
                if ( member.id === req.session.profile.id ) {
                    isUsersAtLeastReporter.user = true;
                } else if ( member.id === Config.gitlab.account.id ) {
                    isUsersAtLeastReporter.dojo = true;
                }
            }
        });

        if ( isUsersAtLeastReporter.user && isUsersAtLeastReporter.dojo ) {
            req.session.sendResponse(res, StatusCodes.OK);
            return true;
        } else {
            req.session.sendResponse(res, StatusCodes.UNAUTHORIZED, undefined, 'Template access unauthorized', DojoStatusCode.GITLAB_TEMPLATE_ACCESS_UNAUTHORIZED);
            return false;
        }
    }

    async protectBranch(repoId: number, branchName: string, allowForcePush: boolean, allowedToMerge: ProtectedBranchAccessLevel, allowedToPush: ProtectedBranchAccessLevel, allowedToUnprotect: ProtectedBranchAccessLevel): Promise<ProtectedBranchSchema> {
        return await this.api.ProtectedBranches.protect(repoId, branchName, {
            allowForcePush      : allowForcePush,
            mergeAccessLevel    : allowedToMerge,
            pushAccessLevel     : allowedToPush,
            unprotectAccessLevel: allowedToUnprotect
        });
    }

    async getRepositoryTree(repoId: number, recursive: boolean = true, branch: string = 'main'): Promise<Array<RepositoryTreeSchema>> {
        return await this.api.Repositories.allRepositoryTrees(repoId, {
            recursive: recursive,
            ref      : branch
        });
    }

    async getFile(repoId: number, filePath: string, branch: string = 'main'): Promise<RepositoryFileExpandedSchema> {
        return await this.api.RepositoryFiles.show(repoId, filePath, branch);
    }

    private async createUpdateFile(create: boolean, repoId: number, filePath: string, fileBase64: string, commitMessage: string, branch: string = 'main', authorName: string = 'Dojo', authorMail: string | undefined = undefined): Promise<RepositoryFileSchema> {
        const gitFunction = create ? this.api.RepositoryFiles.create : this.api.RepositoryFiles.edit;

        return await gitFunction(repoId, filePath, branch, fileBase64, commitMessage, {
            encoding   : 'base64',
            authorName : authorName,
            authorEmail: authorMail
        });
    }

    async createFile(repoId: number, filePath: string, fileBase64: string, commitMessage: string, branch: string = 'main', authorName: string = 'Dojo', authorMail: string | undefined = undefined): Promise<RepositoryFileSchema> {
        return this.createUpdateFile(true, repoId, filePath, fileBase64, commitMessage, branch, authorName, authorMail);
    }

    async updateFile(repoId: number, filePath: string, fileBase64: string, commitMessage: string, branch: string = 'main', authorName: string = 'Dojo', authorMail: string | undefined = undefined): Promise<RepositoryFileSchema> {
        return this.createUpdateFile(false, repoId, filePath, fileBase64, commitMessage, branch, authorName, authorMail);
    }

    async deleteFile(repoId: number, filePath: string, commitMessage: string, branch: string = 'main', authorName: string = 'Dojo', authorMail: string | undefined = undefined): Promise<void> {
        await this.api.RepositoryFiles.remove(repoId, filePath, branch, commitMessage, {
            authorName : authorName,
            authorEmail: authorMail
        });
    }
}


export default new GitlabManager();
