import Toolbox from '../shared/Toolbox';


class Config {
    private static _instance: Config;

    public readonly api: {
        port: number
    };

    public readonly database: {
        type: string, host: string, port: number, user: string, password: string, database: string
    };

    public jwtConfig: {
        secret: string; expiresIn: number;
    };

    public permissions: {
        teachingStaff: Array<string>;
    };

    public gitlab: {
        apiURL: string; urls: Array<string>; account: { id: number; username: string; token: string; }; group: { root: number; templates: number; enonces: number; exercices: number; };
    };

    public enonce: {
        default: { description: string; initReadme: boolean; sharedRunnersEnabled: boolean; visibility: string; wikiEnabled: boolean; template: string };
    };

    public readonly userPasswordLength: number;
    public readonly userPasswordSaltRounds: number;

    private constructor() {
        this.api = {
            port: Number(process.env.API_PORT)
        };

        this.database = {
            type    : process.env.DATABASE_TYPE,
            host    : process.env.DATABASE_HOST,
            port    : Number(process.env.DATABASE_PORT),
            user    : process.env.DATABASE_USER,
            password: process.env.DATABASE_PASSWORD,
            database: process.env.DATABASE_NAME
        };

        this.jwtConfig = {
            secret   : process.env.JWT_SECRET_KEY,
            expiresIn: Number(process.env.SESSION_TIMEOUT)
        };

        this.permissions = {
            teachingStaff: JSON.parse(process.env.ROLES_WITH_TEACHING_STAFF_PERMISSIONS)
        };

        this.gitlab = {
            apiURL : process.env.GITLAB_API_URL,
            urls   : JSON.parse(process.env.GITLAB_URLS || '[]'),
            account: {
                id      : Number(process.env.GITLAB_DOJO_ACCOUNT_ID),
                username: process.env.GITLAB_DOJO_ACCOUNT_USERNAME,
                token   : process.env.GITLAB_DOJO_ACCOUNT_TOKEN
            },
            group  : {
                root     : Number(process.env.GITLAB_GROUP_ROOT_ID),
                templates: Number(process.env.GITLAB_GROUP_TEMPLATES_ID),
                enonces  : Number(process.env.GITLAB_GROUP_ENONCES_ID),
                exercices: Number(process.env.GITLAB_GROUP_EXERCICES_ID)
            }
        };

        this.enonce = {
            default: {
                description         : process.env.ENONCE_DEFAULT_DESCRIPTION,
                initReadme          : Toolbox.strToBool(process.env.ENONCE_DEFAULT_INIT_README),
                sharedRunnersEnabled: Toolbox.strToBool(process.env.ENONCE_DEFAULT_SHARED_RUNNERS_ENABLED),
                visibility          : process.env.ENONCE_DEFAULT_VISIBILITY,
                wikiEnabled         : Toolbox.strToBool(process.env.ENONCE_DEFAULT_WIKI_ENABLED),
                template            : process.env.ENONCE_DEFAULT_TEMPLATE.replace('{{USERNAME}}', this.gitlab.account.username).replace('{{TOKEN}}', this.gitlab.account.token)
            }
        };

        this.userPasswordLength = Number(process.env.USER_PASSWORD_LENGTH);
        this.userPasswordSaltRounds = Number(process.env.USER_PASSWORD_SALT_ROUNDS);


    }

    public static get instance(): Config {
        if ( !Config._instance ) {
            Config._instance = new Config();
        }

        return Config._instance;
    }
}


export default Config.instance;
