import Session     from '../controllers/Session';
import Config      from '../config/Config';
import Toolbox     from '../shared/Toolbox';
import * as bcrypt from 'bcryptjs';
import Model       from './Model';
import db          from '../helpers/DatabaseHelper';


class User extends Model {
    static tableName: string = 'Users';

    userID: number = null;
    userFirstName: string = '';
    userLastName: string = '';
    userMail: string = '';
    userGitlabId: number = -1;
    userRole: string = 'student';
    userDeleted: boolean = false;

    userPassword: string = null;

    unencryptedPassword: string = null; // This value is not set from the db. It's a value that is not null only if we have called createPassword function

    currentSession: Session = null;

    public async toJsonObject(): Promise<Object> {
        const result = {
            'id'             : this.userID,
            'firstName'      : this.userFirstName,
            'lastName'       : this.userLastName,
            'mail'           : this.userMail,
            'gitlabId'       : this.userGitlabId,
            'role'           : this.userRole,
            'isTeachingStaff': this.isTeachingStaff,
            'deleted'        : this.userDeleted
        };

        return result;
    };

    get fullName(): string {
        return this.userLastName.toUpperCase() + ' ' + this.userFirstName;
    }

    get isTeachingStaff(): boolean {
        return Config.permissions.teachingStaff.includes(this.userRole);
    }

    public importFromJsonObject(jsonObject: any) {
        this.userID = jsonObject.id;
        this.userFirstName = jsonObject.firstName;
        this.userLastName = jsonObject.lastName;
        this.userMail = jsonObject.mail;
        this.userGitlabId = jsonObject.gitlabId;
        this.userRole = jsonObject.role;
        this.userDeleted = jsonObject.deleted;
    }

    public generateHashedPassword() {
        this.userPassword = bcrypt.hashSync(this.unencryptedPassword, Config.userPasswordSaltRounds);
    }

    public replacePassword(password: string) {
        this.unencryptedPassword = password;
        this.generateHashedPassword();
    }

    public createPassword() {
        this.unencryptedPassword = Toolbox.randomString(Config.userPasswordLength);
        this.generateHashedPassword();
    }

    public toDb(): any {
        return {
            userFirstName: Toolbox.capitalizeName(this.userFirstName),
            userLastName : Toolbox.capitalizeName(this.userLastName),
            userRole     : this.userRole,
            userMail     : this.userMail,
            userGitlabId : this.userGitlabId,
            userPassword : this.userPassword
        };
    }

    async create(): Promise<User> {
        const id = await db(User.tableName).insert(this.toDb());
        this.userID = id[0];

        return this;
    }

    update(): Promise<void> {
        return db(User.tableName).where('userID', this.userID).update(this.toDb());
    }

    updatePassword(): Promise<void> {
        return db(User.tableName).where('userID', this.userID).update({ 'userPassword': this.userPassword });
    }

    del(): Promise<void> {
        return db(User.tableName).where('userID', this.userID).update({ 'userDeleted': true });
    }
}


export default User;
