import { Express }               from 'express-serve-static-core';
import express                   from 'express';
import * as ExpressValidator     from 'express-validator';
import { StatusCodes }           from 'http-status-codes';
import RoutesManager             from '../express/RoutesManager';
import ParamsValidatorMiddleware from '../middlewares/ParamsValidatorMiddleware';
import ApiRequest                from '../models/ApiRequest';
import SecurityMiddleware        from '../middlewares/SecurityMiddleware';
import SecurityCheckType         from '../types/SecurityCheckType';
import GitlabUser                from '../shared/types/Gitlab/GitlabUser';
import GitlabManager             from '../managers/GitlabManager';
import Config                    from '../config/Config';
import GitlabMember              from '../shared/types/Gitlab/GitlabMember';
import GitlabAccessLevel         from '../shared/types/Gitlab/GitlabAccessLevel';
import GitlabRepository          from '../shared/types/Gitlab/GitlabRepository';
import UserManager               from '../managers/UserManager';
import User                      from '../models/User';
import Enonce                    from '../models/Enonce';
import EnonceStaff               from '../models/EnonceStaff';
import { AxiosError }            from 'axios';
import logger                    from '../shared/logging/WinstonLogger';
import DojoValidators            from '../helpers/DojoValidators';


class EnonceRoutes implements RoutesManager {
    private static _instance: EnonceRoutes;

    private constructor() { }

    public static get instance(): EnonceRoutes {
        if ( !EnonceRoutes._instance ) {
            EnonceRoutes._instance = new EnonceRoutes();
        }

        return EnonceRoutes._instance;
    }

    private readonly enonceValidator: ExpressValidator.Schema = {
        name    : {
            trim    : true,
            notEmpty: true
        },
        members : {
            trim           : true,
            notEmpty       : true,
            customSanitizer: DojoValidators.arraySanitizer
        },
        template: {
            trim           : true,
            custom         : DojoValidators.templateUrlValidator,
            customSanitizer: DojoValidators.templateUrlSanitizer
        }
    };

    registerOnBackend(backend: Express) {
        backend.get('/enonces/:enonceNameOrUrl', SecurityMiddleware.check(true), this.getEnonce);
        backend.post('/enonces', SecurityMiddleware.check(true, SecurityCheckType.TEACHING_STAFF), ParamsValidatorMiddleware.validate(this.enonceValidator), this.createEnonce);
    }

    // Get an enonce by its name or gitlab url
    private async getEnonce(req: ApiRequest, res: express.Response) {
        return req.boundParams.enonce ? req.session.sendResponse(res, StatusCodes.OK, req.boundParams.enonce.toJsonObject()) : res.status(StatusCodes.NOT_FOUND).send();
    }

    private async createEnonce(req: ApiRequest, res: express.Response) {
        const params: { name: string, members: Array<GitlabUser>, template: string } = req.body;

        let repository: GitlabRepository;
        try {
            repository = await GitlabManager.createRepository(params.name, Config.enonce.default.description.replace('{{ENONCE_NAME}}', params.name), Config.enonce.default.visibility, Config.enonce.default.initReadme, Config.gitlab.group.enonces, Config.enonce.default.sharedRunnersEnabled, Config.enonce.default.wikiEnabled, params.template);
        } catch ( error ) {
            if ( error instanceof AxiosError ) {
                if ( error.response.data.message.name && error.response.data.message.name == 'has already been taken' ) {
                    return res.status(StatusCodes.CONFLICT).send();
                }

                return res.status(error.response.status).send();
            }

            logger.error(error);
            return res.status(StatusCodes.INTERNAL_SERVER_ERROR).send();
        }

        try {
            const members: Array<GitlabMember | false> = await Promise.all([ req.session.profile.userGitlabId, ...params.members.map(member => member.id) ].map(async (memberId: number): Promise<GitlabMember | false> => {
                try {
                    return await GitlabManager.addRepositoryMember(repository.id, memberId, GitlabAccessLevel.Maintainer);
                } catch ( e ) {
                    return false;
                }
            }));

            const enonce: Enonce = await Enonce.createFromSql({
                                                                  enonceName              : repository.name,
                                                                  enonceGitlabId          : repository.id,
                                                                  enonceGitlabLink        : repository.web_url,
                                                                  enonceGitlabCreationInfo: JSON.stringify(repository),
                                                                  enonceGitlabLastInfo    : JSON.stringify(repository),
                                                                  enonceGitlabLastInfoTs  : new Date().getTime()
                                                              }).create();

            let dojoUsers: Array<User> = [ req.session.profile, ...(await UserManager.getFromGitlabUsers(params.members, true) as Array<User>) ];
            dojoUsers = dojoUsers.reduce((unique, user) => (unique.findIndex(uniqueUser => uniqueUser.userID === user.userID) !== -1 ? unique : [ ...unique, user ]), Array<User>());
            await Promise.all(dojoUsers.map(dojoUser => EnonceStaff.createFromSql({
                                                                                      enonceName: enonce.enonceName,
                                                                                      userID    : dojoUser.userID
                                                                                  }).create()));

            return req.session.sendResponse(res, StatusCodes.OK, enonce.toJsonObject());
        } catch ( error ) {
            if ( error instanceof AxiosError ) {
                return res.status(error.response.status).send();
            }

            logger.error(error);
            return res.status(StatusCodes.INTERNAL_SERVER_ERROR).send();
        }
    }
}


export default EnonceRoutes.instance;
