import { Express }               from 'express-serve-static-core';
import express                   from 'express';
import * as ExpressValidator     from 'express-validator';
import { StatusCodes }           from 'http-status-codes';
import * as bcrypt               from 'bcryptjs';
import RoutesManager             from '../express/RoutesManager';
import ParamsValidatorMiddleware from '../middlewares/ParamsValidatorMiddleware';
import ApiRequest                from '../models/ApiRequest';
import UserManager               from '../managers/UserManager';
import User                      from '../models/User';
import SecurityMiddleware        from '../middlewares/SecurityMiddleware';


class SessionRoutes implements RoutesManager {
    private static _instance: SessionRoutes;

    private constructor() { }

    public static get instance(): SessionRoutes {
        if ( !SessionRoutes._instance ) {
            SessionRoutes._instance = new SessionRoutes();
        }

        return SessionRoutes._instance;
    }

    private readonly loginValidator: ExpressValidator.Schema = {
        user    : {
            trim    : true,
            notEmpty: true
        },
        password: {
            trim    : true,
            notEmpty: true
        }
    };

    registerOnBackend(backend: Express) {
        backend.post('/login', ParamsValidatorMiddleware.validate(this.loginValidator), this.login);
        backend.get('/test_session', SecurityMiddleware.check(true), (req: ApiRequest, res: express.Response) => req.session.sendResponse(res, StatusCodes.OK));
    }

    private async login(req: ApiRequest, res: express.Response) {
        const params: { user: string, password: string } = req.body;

        const user: User | undefined = await UserManager.getByMail(params.user);

        if ( user ) {
            if ( bcrypt.compareSync(params.password, user.userPassword) ) {
                req.session.profile = user;

                req.session.sendResponse(res, StatusCodes.OK);
                return;
            }
        }

        req.session.sendResponse(res, StatusCodes.NOT_FOUND);
    }
}


export default SessionRoutes.instance;
