import express                        from 'express';
import { TagType }                    from '@prisma/client';
import * as ExpressValidator          from 'express-validator';
import { StatusCodes }                from 'http-status-codes';
import RoutesManager                  from '../express/RoutesManager';
import { Express }                    from 'express-serve-static-core';
import db                             from '../helpers/DatabaseHelper';
import SecurityCheckType              from '../types/SecurityCheckType';
import SecurityMiddleware             from '../middlewares/SecurityMiddleware';
import ParamsValidatorMiddleware      from '../middlewares/ParamsValidatorMiddleware';

class TagRoutes implements RoutesManager {
    private readonly tagsValidatorNameType: ExpressValidator.Schema = {
        name: {
            trim: true,
            notEmpty: true
        },
        type: {
            trim: true,
            notEmpty: true
        }
    };

    private readonly tagsValidatorStatus: ExpressValidator.Schema = {
        status: {
            trim: true,
            notEmpty: true
        },
    };

    private readonly tagsValidatorName: ExpressValidator.Schema = {
        tagProposalName: {
            trim: true,
            notEmpty: true
        },
    };

    registerOnBackend(backend: Express) { 
        backend.post('/tags', SecurityMiddleware.check(true, SecurityCheckType.TEACHING_STAFF), ParamsValidatorMiddleware.validate(this.tagsValidatorNameType), this.addTag.bind(this));
        backend.delete('/tags/:tageName', SecurityMiddleware.check(true, SecurityCheckType.ADMIN), this.deleteTag.bind(this));
        backend.get('/tags/proposals/:state', SecurityMiddleware.check(true, SecurityCheckType.ADMIN), this.getSubmittedTag.bind(this));
        backend.post('/tags/proposals', SecurityMiddleware.check(true, SecurityCheckType.TEACHING_STAFF), ParamsValidatorMiddleware.validate(this.tagsValidatorNameType), this.SubmitTag.bind(this));
        backend.patch('/tags/proposals', SecurityMiddleware.check(true, SecurityCheckType.ADMIN), ParamsValidatorMiddleware.validate(this.tagsValidatorName), this.validateTag.bind(this));
    }

    private async addTag(req: express.Request, res: express.Response) {
        const tagName = req.body.name
        const tagType = req.body.type

        if(tagType != TagType.USERDEFINED && !req.session.profile.isAdmin) {
            return req.session.sendResponse(res, StatusCodes.FORBIDDEN);
        }

        db.tag.upsert({
            where : { name: tagName },
            update: {},
            create: {
                name            : tagName,
                type            : tagType,
            }
        })
        return req.session.sendResponse(res, StatusCodes.OK, {
            tag           : req.body.type,
            name          : req.body.name
        }, "Tag ajouté avec succès");
    }
    private async deleteTag(req: express.Request, res: express.Response) {
        const tagName = req.params.name
        
        db.tag.delete({
            where : { name: tagName }
        })
        return req.session.sendResponse(res, StatusCodes.OK, "Tag supprimé avec succès");
    }
    private async getSubmittedTag(req: express.Request, res: express.Response) {
        const state = req.params.state

        db.submissionTag.findMany({
            where : { 
                state: state 
            }
        })
        return req.session.sendResponse(res, StatusCodes.OK);
    }
    private async SubmitTag(req: express.Request, res: express.Response) {
        const tagName = req.body.name
        const tagType = req.body.type

        db.submissionTag.upsert({
            where : { name: tagName },
            update: {},
            create: {
                name            : tagName,
                type            : tagType,
                state           : "PendingApproval"
            }
        })

        return req.session.sendResponse(res, StatusCodes.OK, {
            name          : req.body.name,
            tag           : req.body.tag
        });        
    }
    private async validateTag(req: express.Request, res: express.Response) {
        const state = req.body.state

        if(state == "PendingApproval"){
            return req.session.sendResponse(res, StatusCodes.OK, "Approbation toujours en attente"); 
        } else if (state == "Declined"){
            const detail = req.body.details
            return req.session.sendResponse(res, StatusCodes.OK, detail); 
        } else{
            const tagName = req.body.tagProposalName
            const tagType = req.body.type
            db.tag.upsert({
                where : { name: tagName },
                update: {},
                create: {
                    name            : tagName,
                    type            : tagType,
                }
            })
        }
        return req.session.sendResponse(res, StatusCodes.OK, "Tag accepté"); 
    }
}

export default new TagRoutes();
