import { TypedEmitter }              from 'tiny-typed-emitter';
import AssignmentValidatorEvents     from '../../types/Dojo/AssignmentValidatorEvents';
import SharedAssignmentHelper        from '../../../shared/helpers/Dojo/SharedAssignmentHelper';
import path                          from 'node:path';
import AssignmentCheckerError        from '../../../shared/types/Dojo/AssignmentCheckerError';
import fs                            from 'fs-extra';
import ClientsSharedConfig           from '../../config/ClientsSharedConfig';
import YAML                          from 'yaml';
import DojoDockerCompose             from '../../types/Dojo/DojoDockerCompose';
import { exec, spawn }               from 'child_process';
import AssignmentFile                from '../../../shared/types/Dojo/AssignmentFile';
import ExerciseDockerCompose         from './ExerciseDockerCompose';
import util                          from 'util';
import Assignment, { Language }      from '../../models/Assignment';
import ClientsSharedAssignmentHelper from './ClientsSharedAssignmentHelper';
import { spawnSync }                 from 'node:child_process';
import SharedConfig                  from '../../../shared/config/SharedConfig';
import { add }                       from 'winston';


const execAsync = util.promisify(exec);


class AssignmentValidator {
    readonly events: TypedEmitter<AssignmentValidatorEvents> = new TypedEmitter<AssignmentValidatorEvents>();

    public displayableLogs: string = '';
    public allLogs: string = '';

    public isFinished: boolean = false;
    public success: boolean = false;
    public exitCode: number = -1;
    public fatalErrorMessage: string = '';

    private currentStep: string = 'NOT_RUNNING';
    private currentSubStep: string = 'NOT_RUNNING';

    constructor(private folderAssignment: string) {
        this.events.on('logs', (log: string, _error: boolean, displayable: boolean) => {
            this.allLogs += log;
            this.displayableLogs += displayable ? log : '';
        });

        this.events.on('finished', (success: boolean, exitCode: number) => {
            this.isFinished = true;
            this.success = success;
            this.exitCode = exitCode;
        });
    }

    private newStep(name: string, message: string) {
        this.currentStep = name;
        this.events.emit('step', name, message);
    }

    private newSubStep(name: string, message: string) {
        this.currentSubStep = name;
        this.events.emit('subStep', name, message);
    }

    private endStep(message: string, error: boolean) {
        this.events.emit('endStep', this.currentStep, message, error);
    }

    private endSubStep(message: string, error: boolean) {
        this.events.emit('endSubStep', this.currentStep, message, error);
    }

    private log(message: string, error: boolean, displayable: boolean) {
        this.events.emit('logs', message, error, displayable, this.currentStep, this.currentSubStep);
    }

    private finished(success: boolean, code: number) {
        this.events.emit('finished', success, code);
    }

    private emitError(subStepMessage: string, stepMessage: string, code: AssignmentCheckerError) {
        this.fatalErrorMessage = stepMessage;

        this.endSubStep(subStepMessage, true);
        this.endStep(stepMessage, true);
        this.finished(false, code);
    }

    run(doDown: boolean = false, runSonar: boolean = false) {
        (async () => {
            let dockerComposeFile: DojoDockerCompose;
            let assignmentFile: AssignmentFile;
            let assignment: Assignment;

            /*
             //////////////////////////////////////////////////////////////////////////////////////////////////////////// Step 1: Check requirements
             -   Check if Docker daemon is running
             -   Check if required files exists
             */
            {
                this.newStep('REQUIREMENTS_CHECKING', 'Please wait while we are checking requirements...');


                // Check requirements
                this.newSubStep('DOCKER_RUNNING', 'Checking if Docker daemon is running');
                try {
                    await execAsync(`docker ps`);
                } catch ( error ) {
                    this.emitError(`Docker daemon isn't running`, `Some requirements are not satisfied.`, AssignmentCheckerError.DOCKER_DAEMON_NOT_RUNNING);
                    return;
                }
                this.endSubStep('Docker daemon is running', false);


                // Check if required files exists
                this.newSubStep('REQUIRED_FILES_EXISTS', 'Checking if required files exists');
                const files = fs.readdirSync(this.folderAssignment);
                const missingFiles = ClientsSharedConfig.assignment.neededFiles.map((file: string): [ string, boolean ] => [ file, files.includes(file) ]).filter((file: [ string, boolean ]) => !file[1]);

                if ( missingFiles.length > 0 ) {
                    this.emitError(`The exercise folder is missing the following files: ${ missingFiles.map((file: [ string, boolean ]) => file[0]).join(', ') }`, 'Some requirements are not satisfied', AssignmentCheckerError.REQUIRED_FILES_MISSING);
                    return;
                }
                this.endSubStep('All required files exists', false);


                this.endStep('All requirements are satisfied', false);
            }

            /*
             //////////////////////////////////////////////////////////////////////////////////////////////////////////// Step 2: Check assignment
             -   Check if assignment exists in backend
             */
            {
                this.newStep('ASSIGNMENT_CHECKING', 'Please wait while we are checking the assignment...');

                const resp = await ClientsSharedAssignmentHelper.getAssignmentByName(ClientsSharedConfig.assignment.name);
                if (resp == undefined) {
                    this.emitError(`The assignment doesn't exist. An assignment must be created with "assignment create" before checking it.`, `Assignment doesn't exists`, AssignmentCheckerError.ASSIGNMENT_MISSING);
                    return;
                } else {
                    assignment = resp;
                }

                this.endStep('Assignment exists', false);
            }


            /*
             //////////////////////////////////////////////////////////////////////////////////////////////////////////// Step 3: dojo_assignment.json file validation
             -   Structure validation
             -   Immutable files validation (Check if exists and if the given type is correct)
             -   Build line validation (for C-derived languages and sonar activated projects)
             */
            {
                this.newStep('ASSIGNMENT_FILE_VALIDATION', 'Please wait while we are validating dojo_assignment.json file...');


                // Structure validation
                this.newSubStep('ASSIGNMENT_FILE_SCHEMA_VALIDATION', 'Validating dojo_assignment.json file schema');
                const validationResults = SharedAssignmentHelper.validateDescriptionFile(path.join(this.folderAssignment, ClientsSharedConfig.assignment.filename));
                if ( !validationResults.isValid ) {
                    this.emitError(`dojo_assignment.json file schema is invalid.\nHere are the errors:\n${ validationResults.error }`, 'dojo_assignment.json file is invalid', AssignmentCheckerError.ASSIGNMENT_FILE_SCHEMA_ERROR);
                    return;
                }
                assignmentFile = validationResults.content!;
                this.endSubStep('dojo_assignment.json file schema is valid', false);


                // Immutable files validation (Check if exists and if the given type is correct)
                this.newSubStep('ASSIGNMENT_FILE_IMMUTABLES_VALIDATION', 'Validating immutable files');
                for ( const immutable of validationResults.content!.immutable ) {
                    const immutablePath = path.join(this.folderAssignment, immutable.path);
                    if ( !fs.existsSync(immutablePath) ) {
                        this.emitError(`Immutable path not found: ${ immutable.path }`, 'dojo_assignment.json file is invalid', AssignmentCheckerError.IMMUTABLE_PATH_NOT_FOUND);
                        return;
                    }

                    const isDirectory = fs.lstatSync(immutablePath).isDirectory();
                    if ( isDirectory && !immutable.isDirectory ) {
                        this.emitError(`Immutable (${ immutable.path }) is declared as a file but is a directory.`, 'dojo_assignment.json file is invalid', AssignmentCheckerError.IMMUTABLE_PATH_IS_NOT_DIRECTORY);
                        return;
                    } else if ( !isDirectory && immutable.isDirectory === true ) {
                        this.emitError(`Immutable (${ immutable.path }) is declared as a directory but is a file.`, 'dojo_assignment.json file is invalid', AssignmentCheckerError.IMMUTABLE_PATH_IS_DIRECTORY);
                        return;
                    }
                }
                this.endSubStep('Immutable files are valid', false);

                // Build line validation (only if language is C/CPP/OBJ-C and sonar activated)
                if ([Language.c, Language.cpp, Language.objc].includes(assignment.language) && assignment.useSonar) {
                    this.newSubStep('ASSIGNMENT_FILE_BUILD_LINE_VALIDATION', 'Validating build line');
                    const build = validationResults.content!.buildLine;
                    if (build == undefined || build.trim() == "") {
                        this.emitError(`BuildLine is required for this language`, 'dojo_assignment.json file is invalid', AssignmentCheckerError.BUILD_LINE_MISSING);
                        return;
                    }
                    this.endSubStep('Build line is valid', false);
                }

                this.endStep('dojo_assignment.json file is valid', false);
            }


            /*
             //////////////////////////////////////////////////////////////////////////////////////////////////////////// Step 4: Docker Compose file validation
             -   Global validation
             -   Validation of the containers and volumes named in dojo_assignment.json
             */
            {
                this.newStep('DOCKER_COMPOSE_VALIDATION', 'Please wait while we are validating docker compose file...');


                // Global validation
                this.newSubStep('DOCKER_COMPOSE_STRUCTURE_VALIDATION', 'Docker compose file structure validation');
                try {
                    dockerComposeFile = YAML.parse(fs.readFileSync(path.join(this.folderAssignment, 'docker-compose.yml'), 'utf8')) as DojoDockerCompose;
                } catch ( error ) {
                    this.emitError(`Docker compose file yaml structure is invalid.`, 'Docker compose file is invalid', AssignmentCheckerError.COMPOSE_FILE_YAML_ERROR);
                    return;
                }

                try {
                    await new Promise<void>((resolve, reject) => {
                        const dockerComposeValidation = spawn(`docker compose -f docker-compose.yml config --quiet`, {
                            cwd  : this.folderAssignment,
                            shell: true
                        });

                        dockerComposeValidation.on('exit', (code) => {
                            code !== null && code == 0 ? resolve() : reject();
                        });
                    });
                } catch ( error ) {
                    this.emitError(`Docker compose file structure is invalid.`, 'Docker compose file is invalid', AssignmentCheckerError.COMPOSE_FILE_SCHEMA_ERROR);
                    return;
                }
                this.endSubStep('Docker compose file structure is valid', false);


                // Validation of the containers and volumes named in dojo_assignment.json
                this.newSubStep('DOCKER_COMPOSE_CONTENT_VALIDATION', 'Docker compose file content validation');
                if ( !(assignmentFile.result.container in dockerComposeFile!.services) ) {
                    this.emitError(`Container specified in ${ ClientsSharedConfig.assignment.filename } is missing from compose file.`, 'Docker compose file is invalid', AssignmentCheckerError.COMPOSE_FILE_CONTAINER_MISSING);
                    return;
                }
                if ( assignmentFile.result.volume && (!dockerComposeFile!.volumes || !(assignmentFile.result.volume in dockerComposeFile!.volumes)) ) {
                    this.emitError(`Volume specified in ${ ClientsSharedConfig.assignment.filename } is missing from compose file.`, 'Docker compose file is invalid', AssignmentCheckerError.COMPOSE_FILE_VOLUME_MISSING);
                    return;
                }
                this.endSubStep('Docker compose file content is valid', false);


                this.endStep('Docker compose file is valid', false);
            }


            /*
             //////////////////////////////////////////////////////////////////////////////////////////////////////////// Step 5: Dockerfiles validation
             -   Check if file exists
             -   TODO - Dockerfile structure linter - Issue #51 - https://github.com/hadolint/hadolint
             */
            {
                this.newStep('DOCKERFILE_VALIDATION', 'Please wait while we are validating dockerfiles...');


                this.newSubStep('DOCKERFILE_EXIST', 'Docker compose file content validation');
                const dockerfilesPaths = Object.values(dockerComposeFile!.services).filter((service) => service.build).map((service) => path.join(this.folderAssignment, service.build!.context ?? '', service.build!.dockerfile!));
                const filesNotFound = dockerfilesPaths.filter((dockerfilePath) => !fs.existsSync(dockerfilePath));
                if ( filesNotFound.length > 0 ) {
                    this.emitError(`Dockerfiles not found: ${ filesNotFound.join(', ') }`, 'Dockerfiles are invalid', AssignmentCheckerError.DOCKERFILE_NOT_FOUND);
                    return;
                }
                this.endSubStep('Docker compose file content is valid', false);


                this.endStep('Dockerfiles are valid', false);
            }


            /*
             //////////////////////////////////////////////////////////////////////////////////////////////////////////// Step 6: Sonar analysis
             - Analyse the project with SonarCube
             */
            if (assignment.useSonar && runSonar)
            {
                this.newStep('ASSIGNMENT_SONAR', 'Please wait while we are running Sonar analysis on the assignment...');

                let additionalParams: string[] = [];

                this.newSubStep('SONAR_BUILD', 'Build files');

                const buildProcess = spawnSync('docker', ['build', '--tag', 'dojo-sonar-scanner', '/sonar']);
                if ( buildProcess.status !== 0 ) {
                    this.emitError(`Build sonar image failed`, 'Sonar analysis failure', AssignmentCheckerError.SONAR_ANALYSIS_FAILED);
                    console.log(buildProcess.stdout.toString())
                    console.log(buildProcess.stderr.toString())
                    return;
                }

                if ([Language.c, Language.cpp, Language.objc].includes(assignment.language) && assignmentFile.buildLine != undefined) {
                    const process = spawnSync('docker run -v ./:/usr/src dojo-sonar-scanner /usr/local/bin/build-wrapper-linux-x86-64 --out-dir bw-output ' + assignmentFile.buildLine, [], { shell: true })
                    if ( process.status !== 0 ) {
                        this.emitError(`Failed to build files using buildLine`, 'Sonar analysis failure', AssignmentCheckerError.SONAR_ANALYSIS_FAILED);
                        console.log(process.stdout.toString())
                        console.log(process.stderr.toString())
                        return;
                    }
                    additionalParams = ['-Dsonar.cfamily.build-wrapper-output=/usr/src/bw-output'];
                }
                this.endSubStep('Sonar files build success', false);

                this.newSubStep('SONAR_RUN', 'Run sonar analysis');


                const process = spawnSync('docker', ['run', '-v', './:/usr/src', 'dojo-sonar-scanner' , 'sonar-scanner', '-Dsonar.qualitygate.wait=true', '-Dsonar.projectKey=' + assignment.sonarKey, '-Dsonar.sources=.', '-Dsonar.host.url=' + SharedConfig.sonar.url, '-Dsonar.login=' + SharedConfig.sonar.token, ...additionalParams])
                if ( process.status !== 0 ) {
                    this.emitError(`Sonar gate failed`, 'Sonar analysis failure', AssignmentCheckerError.SONAR_ANALYSIS_FAILED);
                    return;
                }
                this.endSubStep('Sonar gate passed', false);

                this.endStep('Sonar analysis success', false);
            }


            /*
             //////////////////////////////////////////////////////////////////////////////////////////////////////////// Step 7: Run
             -   Make a run of the assignment (If the return code is 0, the assignment is not valid because it means that there no need of modification for succeed the exercise)
             */
            {
                this.newStep('ASSIGNMENT_RUN', 'Please wait while we are running the assignment...');


                const exerciseDockerCompose = new ExerciseDockerCompose(ClientsSharedConfig.dockerCompose.projectName, assignmentFile, this.folderAssignment);

                try {
                    await new Promise<void>((resolve, reject) => {
                        exerciseDockerCompose.events.on('logs', (log: string, error: boolean, displayable: boolean) => {
                            this.log(log, error, displayable);
                        });

                        exerciseDockerCompose.events.on('step', (name: string, message: string) => {
                            this.newSubStep(name, message);
                        });

                        exerciseDockerCompose.events.on('endStep', (stepName: string, message: string, error: boolean) => {
                            this.endSubStep(message, error);
                        });

                        exerciseDockerCompose.events.on('finished', (success: boolean, exitCode: number) => {
                            exitCode != 0 ? resolve() : reject();
                        });

                        exerciseDockerCompose.run(doDown);
                    });
                } catch ( error ) {
                    this.fatalErrorMessage = 'Assignment is already solved';
                    this.endStep(this.fatalErrorMessage, true);
                    this.finished(false, AssignmentCheckerError.COMPOSE_RUN_SUCCESSFULLY);
                    return;
                }


                this.endStep('Assignment run successfully', false);
            }


            this.finished(true, 0);
        })();
    }
}


export default AssignmentValidator;