import { TypedEmitter }          from 'tiny-typed-emitter';
import AssignmentValidatorEvents from '../../types/Dojo/AssignmentValidatorEvents';
import SharedAssignmentHelper    from '../../../shared/helpers/Dojo/SharedAssignmentHelper';
import path                      from 'node:path';
import AssignmentCheckerError    from '../../../shared/types/Dojo/AssignmentCheckerError';
import fs                        from 'fs-extra';
import ClientsSharedConfig       from '../../config/ClientsSharedConfig';
import YAML                      from 'yaml';
import DojoDockerCompose         from '../../types/Dojo/DojoDockerCompose';
import { exec, spawn }           from 'child_process';
import AssignmentFile            from '../../../shared/types/Dojo/AssignmentFile';
import ExerciseDockerCompose     from './ExerciseDockerCompose';
import util                      from 'util';


const execAsync = util.promisify(exec);


class AssignmentValidator {
    private readonly folderAssignment: string;
    private readonly doDown: boolean;

    readonly events: TypedEmitter<AssignmentValidatorEvents> = new TypedEmitter<AssignmentValidatorEvents>();

    public displayableLogs: string = '';
    public allLogs: string = '';

    public isFinished: boolean = false;
    public success: boolean = false;
    public exitCode: number = -1;
    public fatalErrorMessage: string = '';

    private currentStep: string = 'NOT_RUNNING';
    private currentSubStep: string = 'NOT_RUNNING';

    private dockerComposeFile!: DojoDockerCompose;
    private assignmentFile!: AssignmentFile;

    constructor(folderAssignment: string, doDown: boolean = false) {
        this.folderAssignment = folderAssignment;
        this.doDown = doDown;

        this.events.on('logs', (log: string, _error: boolean, displayable: boolean) => {
            this.allLogs += log;
            this.displayableLogs += displayable ? log : '';
        });

        this.events.on('finished', (success: boolean, exitCode: number) => {
            this.isFinished = true;
            this.success = success;
            this.exitCode = exitCode;
        });
    }

    private newStep(name: string, message: string) {
        this.currentStep = name;
        this.events.emit('step', name, message);
    }

    private newSubStep(name: string, message: string) {
        this.currentSubStep = name;
        this.events.emit('subStep', name, message);
    }

    private endStep(message: string, error: boolean) {
        this.events.emit('endStep', this.currentStep, message, error);
    }

    private endSubStep(message: string, error: boolean) {
        this.events.emit('endSubStep', this.currentStep, message, error);
    }

    private log(message: string, error: boolean, displayable: boolean) {
        this.events.emit('logs', message, error, displayable, this.currentStep, this.currentSubStep);
    }

    private finished(success: boolean, code: number) {
        this.events.emit('finished', success, code);
    }

    private emitError(subStepMessage: string, stepMessage: string, code: AssignmentCheckerError) {
        this.fatalErrorMessage = stepMessage;

        this.endSubStep(subStepMessage, true);
        this.endStep(stepMessage, true);
        this.finished(false, code);
    }

    /**
     * Step 1: Check requirements
     * -   Check if Docker daemon is running
     * -   Check if required files exists
     * @private
     */
    private async checkRequirements() {
        this.newStep('REQUIREMENTS_CHECKING', 'Please wait while we are checking requirements...');


        // Check requirements
        this.newSubStep('DOCKER_RUNNING', 'Checking if Docker daemon is running');
        try {
            await execAsync(`docker ps`);
        } catch ( error ) {
            this.emitError(`Docker daemon isn't running`, `Some requirements are not satisfied.`, AssignmentCheckerError.DOCKER_DAEMON_NOT_RUNNING);
            throw new Error();
        }
        this.endSubStep('Docker daemon is running', false);


        // Check if required files exists
        this.newSubStep('REQUIRED_FILES_EXISTS', 'Checking if required files exists');
        const files = fs.readdirSync(this.folderAssignment);
        const missingFiles = ClientsSharedConfig.assignment.neededFiles.map((file: string): [ string, boolean ] => [ file, files.includes(file) ]).filter((file: [ string, boolean ]) => !file[1]);

        if ( missingFiles.length > 0 ) {
            this.emitError(`The exercise folder is missing the following files: ${ missingFiles.map((file: [ string, boolean ]) => file[0]).join(', ') }`, 'Some requirements are not satisfied', AssignmentCheckerError.REQUIRED_FILES_MISSING);
            throw new Error();
        }
        this.endSubStep('All required files exists', false);


        this.endStep('All requirements are satisfied', false);
    }

    /**
     * Step 2: dojo_assignment.json file validation
     * -   Structure validation
     * -   Immutable files validation (Check if exists and if the given type is correct)
     * @private
     */
    private dojoAssignmentFileValidation() {
        this.newStep('ASSIGNMENT_FILE_VALIDATION', 'Please wait while we are validating dojo_assignment.json file...');

        const assignmentFileValidationError = `${ ClientsSharedConfig.assignment.filename } file is invalid`;

        // Structure validation
        this.newSubStep('ASSIGNMENT_FILE_SCHEMA_VALIDATION', 'Validating dojo_assignment.json file schema');
        const validationResults = SharedAssignmentHelper.validateDescriptionFile(path.join(this.folderAssignment, ClientsSharedConfig.assignment.filename));
        if ( !validationResults.isValid ) {
            this.emitError(`dojo_assignment.json file schema is invalid.\nHere are the errors:\n${ validationResults.error }`, assignmentFileValidationError, AssignmentCheckerError.ASSIGNMENT_FILE_SCHEMA_ERROR);
            throw new Error();
        }
        this.assignmentFile = validationResults.content!;
        this.endSubStep('dojo_assignment.json file schema is valid', false);


        // Immutable files validation (Check if exists and if the given type is correct)
        this.newSubStep('ASSIGNMENT_FILE_IMMUTABLES_VALIDATION', 'Validating immutable files');
        for ( const immutable of validationResults.content!.immutable ) {
            const immutablePath = path.join(this.folderAssignment, immutable.path);
            if ( !fs.existsSync(immutablePath) ) {
                this.emitError(`Immutable path not found: ${ immutable.path }`, assignmentFileValidationError, AssignmentCheckerError.IMMUTABLE_PATH_NOT_FOUND);
                throw new Error();
            }

            const isDirectory = fs.lstatSync(immutablePath).isDirectory();
            if ( isDirectory && !immutable.isDirectory ) {
                this.emitError(`Immutable (${ immutable.path }) is declared as a file but is a directory.`, assignmentFileValidationError, AssignmentCheckerError.IMMUTABLE_PATH_IS_NOT_DIRECTORY);
                throw new Error();
            } else if ( !isDirectory && immutable.isDirectory === true ) {
                this.emitError(`Immutable (${ immutable.path }) is declared as a directory but is a file.`, assignmentFileValidationError, AssignmentCheckerError.IMMUTABLE_PATH_IS_DIRECTORY);
                throw new Error();
            }
        }
        this.endSubStep('Immutable files are valid', false);


        this.endStep('dojo_assignment.json file is valid', false);
    }

    /**
     * Step 3: Docker Compose file validation
     * -   Global validation
     * -   Validation of the containers and volumes named in dojo_assignment.json
     * @private
     */
    private async dockerComposeFileValidation() {
        this.newStep('DOCKER_COMPOSE_VALIDATION', 'Please wait while we are validating docker compose file...');

        const composeFileValidationError = `Docker compose file is invalid`;

        // Global validation
        this.newSubStep('DOCKER_COMPOSE_STRUCTURE_VALIDATION', 'Docker compose file structure validation');
        try {
            this.dockerComposeFile = YAML.parse(fs.readFileSync(path.join(this.folderAssignment, 'docker-compose.yml'), 'utf8')) as DojoDockerCompose;
        } catch ( error ) {
            this.emitError(`Docker compose file yaml structure is invalid.`, composeFileValidationError, AssignmentCheckerError.COMPOSE_FILE_YAML_ERROR);
            throw new Error();
        }

        try {
            await new Promise<void>((resolve, reject) => {
                const dockerComposeValidation = spawn(`docker compose -f docker-compose.yml config --quiet`, {
                    cwd  : this.folderAssignment,
                    shell: true
                });

                dockerComposeValidation.on('exit', code => {
                    code !== null && code === 0 ? resolve() : reject();
                });
            });
        } catch ( error ) {
            this.emitError(`Docker compose file structure is invalid.`, composeFileValidationError, AssignmentCheckerError.COMPOSE_FILE_SCHEMA_ERROR);
            throw new Error();
        }
        this.endSubStep('Docker compose file structure is valid', false);


        // Validation of the containers and volumes named in dojo_assignment.json
        this.newSubStep('DOCKER_COMPOSE_CONTENT_VALIDATION', 'Docker compose file content validation');
        if ( !(this.assignmentFile.result.container in this.dockerComposeFile.services) ) {
            this.emitError(`Container specified in ${ ClientsSharedConfig.assignment.filename } is missing from compose file.`, composeFileValidationError, AssignmentCheckerError.COMPOSE_FILE_CONTAINER_MISSING);
            throw new Error();
        }
        if ( this.assignmentFile.result.volume && (!this.dockerComposeFile.volumes || !(this.assignmentFile.result.volume in this.dockerComposeFile.volumes)) ) {
            this.emitError(`Volume specified in ${ ClientsSharedConfig.assignment.filename } is missing from compose file.`, composeFileValidationError, AssignmentCheckerError.COMPOSE_FILE_VOLUME_MISSING);
            throw new Error();
        }
        this.endSubStep('Docker compose file content is valid', false);


        this.endStep('Docker compose file is valid', false);
    }

    /**
     * Step 4: Dockerfiles validation
     * -   Check if file exists
     * -   TODO - Dockerfile structure linter - Issue #51 - https://github.com/hadolint/hadolint
     * @private
     */
    private dockerfilesValidation() {
        this.newStep('DOCKERFILE_VALIDATION', 'Please wait while we are validating dockerfiles...');


        this.newSubStep('DOCKERFILE_EXIST', 'Docker compose file content validation');
        const dockerfilesPaths = Object.values(this.dockerComposeFile.services).filter(service => service.build).map(service => path.join(this.folderAssignment, service.build!.context ?? '', service.build!.dockerfile!));
        const filesNotFound = dockerfilesPaths.filter(dockerfilePath => !fs.existsSync(dockerfilePath));
        if ( filesNotFound.length > 0 ) {
            this.emitError(`Dockerfiles not found: ${ filesNotFound.join(', ') }`, 'Dockerfiles are invalid', AssignmentCheckerError.DOCKERFILE_NOT_FOUND);
            throw new Error();
        }
        this.endSubStep('Docker compose file content is valid', false);


        this.endStep('Dockerfiles are valid', false);
    }

    /**
     * Step 5: Run
     * -   Make a run of the assignment (If the return code is 0, the assignment is not valid because it means that there no need of modification for succeed the exercise)
     * @private
     */
    private async runAssignment() {
        this.newStep('ASSIGNMENT_RUN', 'Please wait while we are running the assignment...');


        const exerciseDockerCompose = new ExerciseDockerCompose(ClientsSharedConfig.dockerCompose.projectName, this.assignmentFile, this.folderAssignment);

        try {
            await new Promise<void>((resolve, reject) => {
                exerciseDockerCompose.events.on('logs', (log: string, error: boolean, displayable: boolean) => {
                    this.log(log, error, displayable);
                });

                exerciseDockerCompose.events.on('step', (name: string, message: string) => {
                    this.newSubStep(name, message);
                });

                exerciseDockerCompose.events.on('endStep', (_stepName: string, message: string, error: boolean) => {
                    this.endSubStep(message, error);
                });

                exerciseDockerCompose.events.on('finished', (_success: boolean, exitCode: number) => {
                    exitCode !== 0 ? resolve() : reject();
                });

                exerciseDockerCompose.run(this.doDown);
            });
        } catch ( error ) {
            this.fatalErrorMessage = 'Assignment is already solved';
            this.endStep(this.fatalErrorMessage, true);
            this.finished(false, AssignmentCheckerError.COMPOSE_RUN_SUCCESSFULLY);
            throw new Error();
        }


        this.endStep('Assignment run successfully', false);
    }

    run() {
        (async () => {
            try {
                await this.checkRequirements();

                this.dojoAssignmentFileValidation();

                await this.dockerComposeFileValidation();

                this.dockerfilesValidation();

                await this.runAssignment();

                this.finished(true, 0);
            } catch ( error ) {
                return;
            }
        })();
    }
}


export default AssignmentValidator;