import AssignmentFile                     from '../../../shared/types/Dojo/AssignmentFile';
import { TypedEmitter }                   from 'tiny-typed-emitter';
import ExerciseRunningEvents              from '../../types/Dojo/ExerciseRunningEvents';
import { spawn }                          from 'child_process';
import ExerciseCheckerError               from '../../../shared/types/Dojo/ExerciseCheckerError';
import { ChildProcessWithoutNullStreams } from 'node:child_process';


class ExerciseDockerCompose {
    readonly events: TypedEmitter<ExerciseRunningEvents> = new TypedEmitter<ExerciseRunningEvents>();

    public displayableLogs: string = '';
    public allLogs: string = '';

    public isFinished: boolean = false;
    public success: boolean = false;
    public exitCode: number = -1;

    private currentStep: string = 'NOT_RUNNING';

    private readonly projectName: string;
    private readonly assignmentFile: AssignmentFile;
    private readonly executionFolder: string;
    private readonly composeFileOverride: Array<string> = [];

    constructor(projectName: string, assignmentFile: AssignmentFile, executionFolder: string, composeFileOverride: Array<string> = []) {
        this.projectName = projectName;
        this.assignmentFile = assignmentFile;
        this.executionFolder = executionFolder;
        this.composeFileOverride = composeFileOverride;

        this.events.on('logs', (log: string, _error: boolean, displayable: boolean) => {
            this.allLogs += log;
            this.displayableLogs += displayable ? log : '';
        });

        this.events.on('finished', (success: boolean, exitCode: number) => {
            this.isFinished = true;
            this.success = success;
            this.exitCode = exitCode;
        });
    }

    private newStep(name: string, message: string) {
        this.currentStep = name;
        this.events.emit('step', name, message);
    }

    private endStep(message: string, error: boolean) {
        this.events.emit('endStep', this.currentStep, message, error);
    }

    private log(message: string, error: boolean, displayable: boolean) {
        this.events.emit('logs', message, error, displayable, this.currentStep);
    }

    private finished(success: boolean, code: number) {
        this.events.emit('finished', success, code);
    }

    private registerChildProcess(childProcess: ChildProcessWithoutNullStreams, resolve: (value: (number | PromiseLike<number>)) => void, reject: (reason?: unknown) => void, displayable: boolean, rejectIfCodeIsNotZero: boolean) {
        childProcess.stdout.on('data', data => {
            this.log(data.toString(), false, displayable);
        });

        childProcess.stderr.on('data', data => {
            this.log(data.toString(), true, displayable);
        });

        childProcess.on('exit', code => {
            code === null || (rejectIfCodeIsNotZero && code !== 0) ? reject(code) : resolve(code);
        });
    }

    run(doDown: boolean = false) {
        (async () => {
            let containerExitCode: number = -1;

            const filesOverrideArguments = this.composeFileOverride.map(file => `--file "${ file }"`).join(' ');
            const dockerComposeCommand = `docker compose --project-name ${ this.projectName } --progress plain --file docker-compose.yml ${ filesOverrideArguments }`;

            // Run the service
            {
                try {
                    this.newStep('COMPOSE_RUN', 'Running Docker Compose file');

                    containerExitCode = await new Promise<number>((resolve, reject) => {

                        this.log('####################################################### Docker Compose & Main Container Logs #######################################################\n', false, false);

                        const dockerCompose = spawn(`${ dockerComposeCommand } run --build --rm ${ this.assignmentFile.result.container }`, {
                            cwd  : this.executionFolder,
                            shell: true,
                            env  : {
                                'DOCKER_BUILDKIT'  : '1',
                                'BUILDKIT_PROGRESS': 'plain', ...process.env
                            }
                        });

                        this.registerChildProcess(dockerCompose, resolve, reject, true, false);
                    });
                } catch ( error ) {
                    this.endStep(`Error while running the docker compose file`, true);
                    this.finished(false, ExerciseCheckerError.DOCKER_COMPOSE_RUN_ERROR);
                    return;
                }
                this.endStep(`Docker Compose file run successfully`, false);
            }

            // Get linked services logs
            {
                try {
                    this.newStep('COMPOSE_LOGS', 'Linked services logs acquisition');

                    await new Promise<number>((resolve, reject) => {

                        this.log('####################################################### Other Services Logs #######################################################\n', false, false);

                        const dockerCompose = spawn(`${ dockerComposeCommand } logs --timestamps`, {
                            cwd  : this.executionFolder,
                            shell: true
                        });

                        this.registerChildProcess(dockerCompose, resolve, reject, false, true);
                    });
                } catch ( error ) {
                    this.endStep(`Error while getting the linked services logs`, true);
                    this.finished(false, ExerciseCheckerError.DOCKER_COMPOSE_LOGS_ERROR);
                    return;
                }
                this.endStep(`Linked services logs acquired`, false);
            }

            // Remove containers if asked
            {
                if ( doDown ) {
                    try {
                        this.newStep('COMPOSE_DOWN', 'Stopping and removing containers');

                        await new Promise<number>((resolve, reject) => {

                            this.log('####################################################### Stop and remove containers #######################################################\n', false, false);

                            const dockerCompose = spawn(`${ dockerComposeCommand } down --volumes`, {
                                cwd  : this.executionFolder,
                                shell: true
                            });

                            this.registerChildProcess(dockerCompose, resolve, reject, false, true);
                        });
                    } catch ( error ) {
                        this.endStep(`Error while stopping and removing containers`, true);
                        this.finished(false, ExerciseCheckerError.DOCKER_COMPOSE_DOWN_ERROR);
                        return;
                    }
                    this.endStep(`Containers stopped and removed`, false);
                }
            }

            // Remove images if asked
            {
                if ( doDown ) {
                    try {
                        this.newStep('COMPOSE_REMOVE_DANGLING', 'Removing dangling images');

                        await new Promise<number>((resolve, reject) => {

                            this.log('####################################################### Remove dangling images #######################################################\n', false, false);

                            const dockerCompose = spawn(`docker image prune --force`, {
                                cwd  : this.executionFolder,
                                shell: true
                            });

                            this.registerChildProcess(dockerCompose, resolve, reject, false, true);
                        });
                    } catch ( error ) {
                        this.endStep(`Error while removing dangling images`, true);
                        this.finished(false, ExerciseCheckerError.DOCKER_COMPOSE_REMOVE_DANGLING_ERROR);
                        return;
                    }
                    this.endStep(`Dangling images removed`, false);
                }
            }

            this.finished(true, containerExitCode);
        })();
    }
}


export default ExerciseDockerCompose;