import { existsSync, readFileSync } from 'fs';
import { join }                     from 'path';
import chalk                        from 'chalk';
import boxen                        from 'boxen';
import Icon                         from '../../../shared/types/Icon';
import AssignmentValidator          from './AssignmentValidator';
import Assignment                   from '../../models/Assignment';
import axios                        from 'axios';
import DojoBackendResponse          from '../../../shared/types/Dojo/DojoBackendResponse';
import ApiRoute                     from '../../types/Dojo/ApiRoute';
import ClientsSharedConfig          from '../../config/ClientsSharedConfig';


class ClientsSharedAssignmentHelper {
    displayExecutionResults(validator: AssignmentValidator, successMessage: string, Style: { INFO: chalk.Chalk, SUCCESS: chalk.Chalk, FAILURE: chalk.Chalk }) {
        const globalResult = validator.success ? Style.SUCCESS(`${ Icon.SUCCESS } Success`) : Style.FAILURE(`${ Icon.FAILURE } Failure`);
        const finalLogGlobalResult = `${ Style.INFO('Global result') } : ${ globalResult }`;

        const finalLogSuccessMessage = validator.success ? `${ successMessage }` : '';
        const finalLogErrorMessage = !validator.success ? `${ Style.INFO('Error message') } :\n${ Style.FAILURE(validator.fatalErrorMessage) }` : '';

        console.log(boxen(`${ finalLogGlobalResult }\n\n${ finalLogSuccessMessage }${ finalLogErrorMessage }`, {
            title         : 'Results',
            titleAlignment: 'center',
            borderColor   : validator.success ? 'green' : 'red',
            borderStyle   : 'bold',
            margin        : 1,
            padding       : 1,
            textAlignment : 'left'
        }));
    }

    private async getAssignment(nameOrUrl: string): Promise<Assignment | undefined> {
        try {
            return (await axios.get<DojoBackendResponse<Assignment>>(`${ ClientsSharedConfig.apiURL }${ ApiRoute.ASSIGNMENT_GET }`.replace('{{nameOrUrl}}', encodeURIComponent(nameOrUrl)))).data.data;
        } catch ( error ) {
            console.log(error);
            return undefined;
        }
    }

    private async extractOriginUrl(content: string): Promise<string> {
        const regexp = /\[remote "origin"]\r?\n\s*url\s*=\s*(.*)\s*\n/gm;
        return Array.from(content.matchAll(regexp), m => m[1])[0];
    }

    async getAssignmentByName(name: string): Promise<Assignment | undefined> {
        return await this.getAssignment(name);
    }

    async getAssignmentFromPath(path: string): Promise<Assignment | undefined> {
        const fullPath = join(path, "./.git/config");
        if (!existsSync(fullPath)) {
            return undefined;
        }
        const content = readFileSync(fullPath, 'utf-8');
        const url = await this.extractOriginUrl(content);
        return await this.getAssignment(url);
    }
}


export default new ClientsSharedAssignmentHelper();