import fs   from 'node:fs';
import path from 'node:path';


export interface RecursiveReaddirFilesOptions {
    /**
     * Ignore files
     * @example `/\/(node_modules|\.git)/`
     */
    ignored?: RegExp;
    /**
     * Specifies a list of `glob` patterns that match files to be included in compilation.
     * @example `/(\.json)$/`
     */
    include?: RegExp;
    /**
     * Specifies a list of files to be excluded from compilation.
     * @example `/(package\.json)$/`
     */
    exclude?: RegExp;
    /** Provide filtering methods to filter data. */
    filter?: (item: IFileDirStat) => boolean;
    /** Do not give the absolute path but the relative one from the root folder */
    replacePathByRelativeOne?: boolean;
    /** Remove stats that are not necessary for transfert */
    liteStats?: boolean;
}


export interface IFileDirStat extends Partial<fs.Stats> {
    /**
     * @example `/a/sum.jpg` => `sum.jpg`
     */
    name: string;
    /**
     * @example `/basic/src/utils/sum.ts`
     */
    path: string;
    /**
     * @example `/a/b.jpg` => `jpg`
     */
    ext?: string;
}


type Callback = (filepath: string, stat: IFileDirStat) => void;


class RecursiveFilesStats {
    async explore(rootPath: string, options: RecursiveReaddirFilesOptions = {}, callback?: Callback): Promise<IFileDirStat[]> {
        return this.getFiles(`${ path.resolve(rootPath) }/`, rootPath, options, [], callback);
    }

    private async getFilesDirsStat(rootPath: string, options: RecursiveReaddirFilesOptions): Promise<Array<IFileDirStat>> {
        const filesData = await fs.promises.readdir(rootPath);
        return filesData.map(file => ({
            name: file,
            path: path.join(rootPath, file)
        })).filter(item => {
            if ( options.include && options.include.test(item.path) ) {
                return true;
            }
            if ( options.exclude && options.exclude.test(item.path) ) {
                return false;
            }
            if ( options.ignored ) {
                return !options.ignored.test(item.path);
            }
            return true;
        });
    }

    private async getFiles(absoluteBasePath: string, rootPath: string, options: RecursiveReaddirFilesOptions = {}, files: IFileDirStat[] = [], callback?: Callback): Promise<IFileDirStat[]> {
        const fileDir: Array<IFileDirStat> = await this.getFilesDirsStat(rootPath, options);

        if ( callback ) {
            fileDir.forEach(item => {
                this.getStat(item.path, absoluteBasePath, options).then(stat => {
                    if ( stat.isDirectory!() ) {
                        this.getFiles(absoluteBasePath, item.path, options, [], callback).then();
                    }
                    callback(item.path, stat);
                });
            });
        } else {
            await Promise.all(fileDir.map(async (item: IFileDirStat) => {
                const stat = await this.getStat(item.path, absoluteBasePath, options);
                if ( stat.isDirectory!() ) {
                    const arr = await this.getFiles(absoluteBasePath, item.path, options, []);
                    files = files.concat(arr);
                } else if ( stat.isFile!() ) {
                    files.push(stat);
                }
            }));
        }
        return files.filter(item => {
            if ( options.filter && typeof options.filter === 'function' ) {
                return options.filter(item);
            }
            return true;
        });
    }

    private async getStat(filepath: string, absoluteRootPath: string, options: RecursiveReaddirFilesOptions): Promise<IFileDirStat> {
        const stat = (await fs.promises.stat(filepath)) as IFileDirStat;
        stat.ext = '';
        if ( stat.isFile!() ) {
            stat.ext = this.getExt(filepath);
            stat.name = path.basename(filepath);
            stat.path = path.resolve(filepath);
        }

        if ( options.replacePathByRelativeOne && stat.path ) {
            stat.path = stat.path.replace(absoluteRootPath, '');
        }

        if ( options.liteStats ) {
            delete stat.dev;
            delete stat.nlink;
            delete stat.uid;
            delete stat.gid;
            delete stat.rdev;
            delete stat.blksize;
            delete stat.ino;
            delete stat.blocks;
            delete stat.atimeMs;
            delete stat.mtimeMs;
            delete stat.ctimeMs;
            delete stat.birthtimeMs;
            delete stat.atime;
            delete stat.ctime;
        }

        return stat;
    }

    /**
     * Get ext
     * @param {String} filePath `/a/b.jpg` => `jpg`
     */
    private getExt(filePath: string): string {
        return path.extname(filePath).replace(/^\./, '').toLowerCase();
    }
}


export default new RecursiveFilesStats();
