import axios                                                                   from 'axios';
import * as GitlabCore                                                         from '@gitbeaker/core';
import { GitbeakerRequestError }                                               from '@gitbeaker/requester-utils';
import { Gitlab, PipelineSchema, ProjectSchema, SimpleUserSchema, UserSchema } from '@gitbeaker/rest';
import GitlabToken                                                             from '../types/Gitlab/GitlabToken.js';


class SharedGitlabManager {
    protected api!: GitlabCore.Gitlab<false>;
    private readonly refreshTokenFunction?: () => Promise<string>;

    setToken(token: string) {
        this.api = new Gitlab(Object.assign({
                                                host: this.gitlabUrl ?? ''
                                            }, this.refreshTokenFunction ? { oauthToken: token } : { token: token }));
    }

    constructor(public gitlabUrl: string, token: string, public clientId?: string, public urlRedirect?: string, public urlToken?: string, refreshTokenFunction?: () => Promise<string>) {
        this.refreshTokenFunction = refreshTokenFunction;
        this.setToken(token);
    }

    protected async executeGitlabRequest<T>(request: () => Promise<T>, refreshTokenIfNeeded: boolean = true): Promise<T> {
        try {
            return await request();
        } catch ( error ) {
            if ( this.refreshTokenFunction && refreshTokenIfNeeded && error instanceof GitbeakerRequestError ) {
                this.setToken(await this.refreshTokenFunction());

                return this.executeGitlabRequest(request, false);
            } else {
                throw error;
            }
        }
    }

    async getTokens(codeOrRefresh: string, isRefresh: boolean = false, clientSecret: string = ''): Promise<GitlabToken> {
        if ( !this.urlToken ) {
            throw new Error('Error when initializing GitlabManager');
        }

        const response = await axios.post<GitlabToken>(this.urlToken, {
            client_id    : this.clientId,
            client_secret: clientSecret,
            grant_type   : isRefresh ? 'refresh_token' : 'authorization_code',
            refresh_token: codeOrRefresh,
            code         : codeOrRefresh,
            redirect_uri : this.urlRedirect
        });

        return response.data;
    }

    public async getUserById(id: number): Promise<UserSchema | undefined> {
        try {
            return await this.executeGitlabRequest(async () => {
                const user = await this.api.Users.show(id);

                return user.id === id ? user : undefined;
            });
        } catch ( e ) {
            return undefined;
        }
    }

    public async getUserByUsername(username: string): Promise<SimpleUserSchema | undefined> {
        try {
            return await this.executeGitlabRequest(async () => {
                const user = await this.api.Users.all({
                                                          username: username,
                                                          maxPages: 1,
                                                          perPage : 1
                                                      });


                return user.length > 0 && user[0].username === username ? user[0] : undefined;
            });
        } catch ( e ) {
            return undefined;
        }
    }

    async getRepository(projectIdOrNamespace: string): Promise<ProjectSchema> {
        return this.executeGitlabRequest(() => this.api.Projects.show(projectIdOrNamespace));
    }

    async getRepositoryPipelines(repoId: number, branch: string = 'main'): Promise<Array<PipelineSchema>> {
        return this.executeGitlabRequest(() => this.api.Pipelines.all(repoId, { ref: branch }));
    }
}


export default SharedGitlabManager;
