import Ajv, { ErrorObject, JTDSchemaType } from 'ajv/dist/jtd';
import fs                                  from 'fs';
import JSON5                               from 'json5';
import AssignmentFile                      from '../../types/Dojo/AssignmentFile';
import GitlabPipelineStatus                from '../../types/Gitlab/GitlabPipelineStatus';
import DojoStatusCode                      from '../../types/Dojo/DojoStatusCode';
import GitlabPipeline                      from '../../types/Gitlab/GitlabPipeline';
import SharedGitlabManager                 from '../../managers/SharedGitlabManager';


class SharedAssignmentHelper {
    private validateDescriptionFileV1(resultsFilePathOrStr: string, isFile: boolean = true): { results: AssignmentFile | undefined, isValid: boolean, errors: Array<ErrorObject | string> | null | undefined } {
        const ajv = new Ajv();

        const schema: JTDSchemaType<AssignmentFile> = {
            properties          : {
                dojoAssignmentVersion: { type: 'uint32' },
                version              : { type: 'uint32' },

                immutable: {
                    elements: {
                        properties        : {
                            path: { type: 'string' }
                        },
                        optionalProperties: {
                            description: { type: 'string' },
                            isDirectory: { type: 'boolean' }
                        }
                    }
                },

                result: {
                    properties        : {
                        container: { type: 'string' }
                    },
                    optionalProperties: {
                        volume: { type: 'string' }
                    }
                }
            },
            additionalProperties: false
        };

        const validator = ajv.compile(schema);

        try {
            const results = JSON5.parse(isFile ? fs.readFileSync(resultsFilePathOrStr, 'utf8') : resultsFilePathOrStr);
            const isValid = validator(results);

            return {
                results: isValid ? results : results as any,
                isValid: isValid,
                errors : validator.errors
            };
        } catch ( error ) {
            return {
                results: undefined,
                isValid: false,
                errors : [ `JSON5 invalid : ${ JSON.stringify(error) }` ]
            };
        }
    }

    validateDescriptionFile(resultsFilePathOrStr: string, isFile: boolean = true, version: number = 1): { results: AssignmentFile | undefined, isValid: boolean, errors: Array<ErrorObject | string> | null | undefined } {
        switch ( version ) {
            case 1:
                return this.validateDescriptionFileV1(resultsFilePathOrStr, isFile);
            default:
                return {
                    results: undefined,
                    isValid: false,
                    errors : [ `Version ${ version } not supported` ]
                };
        }
    }

    async isPublishable(repositoryId: number): Promise<{ isPublishable: boolean, lastPipeline: GitlabPipeline | null, status?: { code: DojoStatusCode, message: string } }> {
        const pipelines = await SharedGitlabManager.getRepositoryPipelines(repositoryId, 'main');
        if ( pipelines.length > 0 ) {
            const lastPipeline = pipelines[0];
            if ( lastPipeline.status != GitlabPipelineStatus.SUCCESS ) {
                return {
                    isPublishable: false,
                    lastPipeline : lastPipeline,
                    status       : {
                        code   : DojoStatusCode.ASSIGNMENT_PUBLISH_PIPELINE_FAILED,
                        message: `Last pipeline status is not "${ GitlabPipelineStatus.SUCCESS }" but "${ lastPipeline.status }".`
                    }
                };
            } else {
                return {
                    isPublishable: true,
                    lastPipeline : lastPipeline
                };
            }
        } else {
            return {
                isPublishable: false,
                lastPipeline : null,
                status       : {
                    code   : DojoStatusCode.ASSIGNMENT_PUBLISH_NO_PIPELINE,
                    message: 'No pipeline found for this assignment.'
                }
            };
        }
    }
}


export default new SharedAssignmentHelper();