import fs           from 'node:fs';
import path         from 'node:path';
import tar          from 'tar-stream';
import stream       from 'node:stream';
import { Writable } from 'stream';
import zlib         from 'zlib';


class ArchiveHelper {
    private async explore(absoluteBasePath: string, rootPath: string, pack: tar.Pack) {
        for ( const file of await fs.promises.readdir(rootPath) ) {
            let filename = file;
            if ( filename !== 'output.tar' ) {
                filename = path.join(rootPath, filename);
                const stat = await fs.promises.stat(filename);
                if ( stat.isDirectory() ) {
                    await this.explore(absoluteBasePath, filename, pack);
                } else {
                    const entry = pack.entry({
                                                 name: filename.replace(absoluteBasePath, ''),
                                                 size: stat.size
                                             }, err => {
                        if ( err ) {
                            throw err;
                        }
                    });
                    const readStream = fs.createReadStream(filename);
                    readStream.pipe(entry);
                }
            }
        }
    }

    private async compress(folderPath: string, tarDataStream: stream.Writable) {
        const pack = tar.pack();

        await this.explore(folderPath, folderPath, pack);

        pack.pipe(zlib.createGzip()).pipe(tarDataStream);
        pack.finalize();
    }

    public async getBase64(folderPath: string): Promise<string> {
        let data: string;
        const tarDataStream = new stream.Writable({
                                                      write(this: Writable, chunk: Buffer, _encoding: BufferEncoding, next: (error?: Error | null) => void) {
                                                          if ( data ) {
                                                              data += chunk.toString('hex');
                                                          } else {
                                                              data = chunk.toString('hex');
                                                          }
                                                          next();
                                                      }
                                                  });

        await this.compress(folderPath, tarDataStream);

        data = await (new Promise(resolve => {
            tarDataStream.on('close', () => {
                resolve(data);
            });
        }));

        return Buffer.from(data, 'hex').toString('base64');
    }
}


export default new ArchiveHelper();