import CommanderCommand   from '../../../../CommanderCommand.js';
import ora                from 'ora';
import DojoBackendManager from '../../../../../managers/DojoBackendManager.js';
import Assignment         from '../../../../../sharedByClients/models/Assignment.js';
import TextStyle          from '../../../../../types/TextStyle.js';
import GlobalHelper       from '../../../../../helpers/GlobalHelper.js';


abstract class AssignmentCorrectionLinkUpdateCommand extends CommanderCommand {
    protected abstract isUpdate: boolean;

    protected defineCommand() {
        this.command
            .description(this.isUpdate ? 'update a correction of an assignment' : 'link an exercise repo as a correction for an assignment')
            .argument('<string>', 'id or url of the exercise that is the correction')
            .requiredOption('-a, --assignment <string>', 'id or url of the assignment of the correction')
            .option('-c, --commit <string>', 'specific commit to link as correction (default: last commit)')
            .option('-d, --description <string>', 'description of the correction (limited to 80 characters, default: empty)')
            .action(this.commandAction.bind(this));
    }

    protected async commandAction(exerciseIdOrUrl: string, options: { assignment: string, commit?: string, description?: string }): Promise<void> {
        let assignment!: Assignment | undefined;

        // Check requirements
        {
            if ( options.description && options.description.length > 80 ) {
                ora('Description is limited to 80 characters').start().fail();
                return;
            }
        }

        // Check access
        {
            assignment = await GlobalHelper.checkAssignmentCorrectionAccess(options.assignment);
            if ( !assignment ) {
                return;
            }


            const assignmentPublishedSpinner: ora.Ora = ora('Checking assignment').start();
            if ( !assignment.published ) {
                assignmentPublishedSpinner.fail(`Assignment is not published`);
                return;
            }
            assignmentPublishedSpinner.succeed(`Assignment is published`);
        }

        // Link the exercise
        {
            console.log(TextStyle.BLOCK('Please wait while we link the exercise...'));

            await DojoBackendManager.linkUpdateCorrection(exerciseIdOrUrl, assignment, options.commit, options.description, this.isUpdate);
        }
    }
}


export default AssignmentCorrectionLinkUpdateCommand;