import CommanderCommand              from '../../CommanderCommand';
import Config                        from '../../../config/Config';
import ora                           from 'ora';
import chalk                         from 'chalk';
import AssignmentValidator           from '../../../sharedByClients/helpers/Dojo/AssignmentValidator';
import ClientsSharedAssignmentHelper from '../../../sharedByClients/helpers/Dojo/ClientsSharedAssignmentHelper';
import SharedConfig                  from '../../../shared/config/SharedConfig';
import GlobalHelper                  from '../../../helpers/GlobalHelper';


class AssignmentCheckCommand extends CommanderCommand {
    protected commandName: string = 'check';
    protected currentSpinner: ora.Ora = ora();
    private verbose: boolean = false;
    private superVerbose: boolean = false;
    private buildPhase: boolean | undefined = undefined;

    protected defineCommand() {
        GlobalHelper.runCommandDefinition(this.command)
            .description('locally run a check of an assignment')
            .action(this.commandAction.bind(this));
    }

    private logsEvent(log: string, _error: boolean, displayable: boolean, _currentStep: string, currentSubStep: string) {
        for ( const line of log.split('\n') ) {
            if ( displayable && this.buildPhase === undefined && line.startsWith('#') ) {
                this.buildPhase = true;
            }

            if ( currentSubStep === 'COMPOSE_RUN' && this.buildPhase === true && line !== '' && !line.startsWith('#') ) {
                this.buildPhase = false;
            }

            if ( SharedConfig.debug || (displayable && (this.superVerbose || this.buildPhase === false)) ) {
                console.log(line);
            }
        }
    }

    private subStepEvent(name: string, message: string) {
        this.currentSpinner = ora({
                                      text  : message,
                                      indent: 4
                                  }).start();

        if ( this.verbose && name === 'COMPOSE_RUN' ) {
            this.currentSpinner.info();
        }
    }

    private endSubStepEvent(stepName: string, message: string, error: boolean) {
        if ( error ) {
            if ( this.verbose && stepName === 'COMPOSE_RUN' ) {
                ora({
                        text  : message,
                        indent: 4
                    }).start().fail();
            } else {
                this.currentSpinner.fail(message);
            }
        } else {
            if ( this.verbose && stepName === 'COMPOSE_RUN' ) {
                ora({
                        text  : message,
                        indent: 4
                    }).start().succeed();
            } else {
                this.currentSpinner.succeed(message);
            }
        }
    }

    protected async commandAction(options: { path: string, verbose: boolean, superVerbose: boolean }): Promise<void> {
        this.superVerbose = options.superVerbose;
        this.verbose = options.verbose || options.superVerbose || SharedConfig.debug;

        const localExercisePath: string = options.path ?? Config.folders.defaultLocalExercise;

        const assignmentValidator = new AssignmentValidator(localExercisePath);

        try {
            await new Promise<void>((resolve, reject) => {
                if ( this.verbose ) {
                    assignmentValidator.events.on('logs', this.logsEvent.bind(this));
                }

                assignmentValidator.events.on('step', (_name: string, message: string) => console.log(chalk.cyan(message)));

                assignmentValidator.events.on('subStep', this.subStepEvent.bind(this));

                assignmentValidator.events.on('endSubStep', this.endSubStepEvent.bind(this));

                assignmentValidator.events.on('finished', (success: boolean) => success ? resolve() : reject());

                assignmentValidator.run();
            });
        } catch ( error ) { /* empty */ }

        ClientsSharedAssignmentHelper.displayExecutionResults(assignmentValidator, `The assignment is ready to be pushed.`, {
            INFO   : chalk.bold,
            SUCCESS: chalk.green,
            FAILURE: chalk.red
        });
    }
}


export default new AssignmentCheckCommand();