import CommanderCommand   from '../../../../CommanderCommand';
import ora                from 'ora';
import DojoBackendManager from '../../../../../managers/DojoBackendManager';
import SessionManager     from '../../../../../managers/SessionManager';
import Assignment         from '../../../../../sharedByClients/models/Assignment';
import TextStyle          from '../../../../../types/TextStyle';


abstract class AssignmentCorrectionLinkUpdateCommand extends CommanderCommand {
    protected abstract isUpdate: boolean;

    protected defineCommand() {
        this.command
        .description(this.isUpdate ? 'update a correction of an assignment' : 'link an exercise repo as a correction for an assignment')
        .argument('<string>', 'id or url of the exercise that is the correction')
        .requiredOption('-a, --assignment <string>', 'id or url of the assignment of the correction')
        .action(this.commandAction.bind(this));
    }

    protected async commandAction(exerciseIdOrUrl: string, options: { assignment: string }): Promise<void> {
        let assignment!: Assignment | undefined;

        // Check access
        {
            console.log(TextStyle.BLOCK('Please wait while we check access...'));

            const assignmentGetSpinner: ora.Ora = ora('Checking if assignment exists').start();
            assignment = await DojoBackendManager.getAssignment(options.assignment);
            if ( !assignment ) {
                assignmentGetSpinner.fail(`The assignment doesn't exists`);
                return;
            }
            assignmentGetSpinner.succeed(`The assignment exists`);


            const assignmentAccessSpinner: ora.Ora = ora('Checking assignment access').start();
            if ( assignment.staff.find(staff => staff.id === SessionManager.profile?.id) === undefined ) {
                assignmentAccessSpinner.fail(`You are not in the staff of the assignment`);
                return;
            }
            assignmentAccessSpinner.succeed(`You are in the staff of the assignment`);


            const assignmentPublishedSpinner: ora.Ora = ora('Checking assignment').start();
            if ( !assignment.published ) {
                assignmentPublishedSpinner.fail(`Assignment is not published`);
                return;
            }
            assignmentPublishedSpinner.succeed(`Assignment is published`);
        }

        // Link the exercise
        {
            console.log(TextStyle.BLOCK('Please wait while we link the exercise...'));

            await DojoBackendManager.linkUpdateCorrection(exerciseIdOrUrl, assignment, this.isUpdate);
        }
    }
}


export default AssignmentCorrectionLinkUpdateCommand;