import axios, { AxiosError, AxiosRequestHeaders } from 'axios';
import SessionManager                             from './SessionManager.js';
import FormData                                   from 'form-data';
import { StatusCodes }                            from 'http-status-codes';
import ClientsSharedConfig                        from '../sharedByClients/config/ClientsSharedConfig.js';
import { version }                                from '../config/Version.js';
import DojoBackendResponse                        from '../shared/types/Dojo/DojoBackendResponse.js';
import DojoStatusCode                             from '../shared/types/Dojo/DojoStatusCode.js';
import boxen                                      from 'boxen';
import { stateConfigFile }                        from '../config/ConfigFiles.js';
import TextStyle                                  from '../types/TextStyle.js';


class HttpManager {
    public handleAuthorizationCommandErrors: boolean = true;

    registerAxiosInterceptor() {
        this.registerRequestInterceptor();
        this.registerResponseInterceptor();
    }

    private registerRequestInterceptor() {
        axios.interceptors.request.use(config => {
            if ( config.data instanceof FormData ) {
                config.headers = { ...config.headers, ...config.data.getHeaders() } as AxiosRequestHeaders;
            }

            if ( config.url && (config.url.indexOf(ClientsSharedConfig.apiURL) !== -1) ) {


                config.headers['Accept-Encoding'] = 'gzip';

                if ( config.data && Object.keys(config.data as { [key: string]: unknown }).length > 0 ) {
                    config.headers['Content-Type'] = 'multipart/form-data';
                }

                if ( SessionManager.isLogged ) {
                    config.headers.Authorization = `Bearer ${ SessionManager.apiToken }`;
                }

                config.headers['client'] = 'DojoCLI';
                config.headers['client-version'] = version;
            }

            return config;
        });
    }

    private requestError(message: string) {
        console.log(boxen(message, {
            title         : 'Request error',
            titleAlignment: 'center',
            borderColor   : 'red',
            borderStyle   : 'bold',
            margin        : 1,
            padding       : 1,
            textAlignment : 'left'
        }));
        process.exit(1);
    }

    private apiMethodNotAllowed(error: AxiosError, isFromApi: boolean) {
        if ( error.response?.status === StatusCodes.METHOD_NOT_ALLOWED && isFromApi && error.response.data ) {
            const data: DojoBackendResponse<void> = error.response.data as DojoBackendResponse<void>;

            switch ( data.code ) {
                case DojoStatusCode.CLIENT_NOT_SUPPORTED.valueOf():
                    this.requestError('Client not recognized by the server. Please contact the administrator.');
                    break;
                case DojoStatusCode.CLIENT_VERSION_NOT_SUPPORTED.valueOf():
                    this.requestError(`CLI version not anymore supported by the server. Please update the CLI by executing this command:\n${ TextStyle.CODE(' dojo upgrade ') }`);
                    break;
                default:
                    break;
            }
        }
    }

    private apiAuthorizationError(error: AxiosError, isFromApi: boolean) {
        if ( this.handleAuthorizationCommandErrors && isFromApi && error.response ) {
            const errorCustomCode = (error.response.data as DojoBackendResponse<unknown> | undefined)?.code ?? error.response.status;

            if ( errorCustomCode === error.response.status ) {
                switch ( error.response.status ) {
                    case StatusCodes.UNAUTHORIZED.valueOf():
                        this.requestError('Session expired or does not exist. Please login again.');
                        break;
                    case StatusCodes.FORBIDDEN.valueOf():
                        this.requestError('Forbidden access.');
                        break;
                    default:
                        this.requestError('Unknown error.');
                        break;
                }
            }
        } else {
            this.handleAuthorizationCommandErrors = true;
        }
    }

    private registerResponseInterceptor() {
        axios.interceptors.response.use(response => {
            if ( response.data && response.data.sessionToken ) {
                SessionManager.apiToken = response.data.sessionToken;
            }

            if ( response.headers['dojocli-latest-version'] ) {
                const latestDojoCliVersion = response.headers['dojocli-latest-version'];
                const storedLatestDojoCliVersion = stateConfigFile.getParam('latestDojoCliVersion') as string | null || '0.0.0';

                if ( latestDojoCliVersion !== storedLatestDojoCliVersion ) {
                    stateConfigFile.setParam('latestDojoCliVersion', latestDojoCliVersion);
                    stateConfigFile.setParam('latestDojoCliVersionNotification', 0);
                }
            }

            return response;
        }, async error => {
            if ( error instanceof AxiosError && error.response ) {
                const isFromApi = error.response.config.url !== undefined && error.response.config.url.indexOf(ClientsSharedConfig.apiURL) !== -1;

                this.apiMethodNotAllowed(error, isFromApi);
                this.apiAuthorizationError(error, isFromApi);
            } else {
                this.requestError('Error connecting to the server. Please check your internet connection. If the problem persists, please contact the administrator.');
            }

            return Promise.reject(error);
        });
    }
}


export default new HttpManager();

