-- Table Département
CREATE TABLE IF NOT EXISTS Departement (
    id_dept INTEGER PRIMARY KEY,
    nom TEXT NOT NULL
);

-- Table Enseignant
CREATE TABLE IF NOT EXISTS Enseignant (
    id_enseignant INTEGER PRIMARY KEY,
    nom TEXT NOT NULL,
    prenom TEXT NOT NULL,
    id_dept INTEGER,
    specialite TEXT,
    FOREIGN KEY (id_dept) REFERENCES Departement(id_dept)
);

-- Table Cours
CREATE TABLE IF NOT EXISTS Cours (
    id_cours INTEGER PRIMARY KEY,
    nom_cours TEXT NOT NULL,
    id_enseignant INTEGER,
    credit INTEGER,
    FOREIGN KEY (id_enseignant) REFERENCES Enseignant(id_enseignant)
);

-- Table Etudiant
CREATE TABLE IF NOT EXISTS Etudiant (
    id_etudiant INTEGER PRIMARY KEY,
    nom TEXT NOT NULL,
    prenom TEXT NOT NULL,
    date_naissance DATE,
    id_dept INTEGER,
    FOREIGN KEY (id_dept) REFERENCES Departement(id_dept)
);

-- Table Inscription
CREATE TABLE IF NOT EXISTS Inscription (
    id_etudiant INTEGER,
    id_cours INTEGER,
    note FLOAT,
    PRIMARY KEY (id_etudiant, id_cours),
    FOREIGN KEY (id_etudiant) REFERENCES Etudiant(id_etudiant),
    FOREIGN KEY (id_cours) REFERENCES Cours(id_cours)
);

-- Table Salle
CREATE TABLE IF NOT EXISTS Salle (
    id_salle INTEGER PRIMARY KEY,
    nom_salle TEXT,
    capacite INTEGER
);

-- Table Séance de cours (planification)
CREATE TABLE IF NOT EXISTS Seance (
    id_seance INTEGER PRIMARY KEY,
    id_cours INTEGER,
    id_salle INTEGER,
    date_seance DATE,
    heure_debut TIME,
    heure_fin TIME,
    FOREIGN KEY (id_cours) REFERENCES Cours(id_cours),
    FOREIGN KEY (id_salle) REFERENCES Salle(id_salle)
);

-- Trigger pour empêcher le chevauchement des séances dans une salle
CREATE TRIGGER IF NOT EXISTS verifier_disponibilite_salle
BEFORE INSERT ON Seance
FOR EACH ROW
BEGIN
    SELECT RAISE(ABORT, 'La salle est déjà réservée à ce moment')
    WHERE EXISTS (
        SELECT 1 FROM Seance
        WHERE id_salle = NEW.id_salle
        AND date_seance = NEW.date_seance
        AND (
            (heure_debut <= NEW.heure_debut AND heure_fin > NEW.heure_debut) OR
            (heure_debut < NEW.heure_fin AND heure_fin >= NEW.heure_fin) OR
            (heure_debut >= NEW.heure_debut AND heure_fin <= NEW.heure_fin)
        )
    );
END;

-- ============================================================================
-- DATA POPULATION
-- ============================================================================

-- Insertion Départements
INSERT INTO Departement (id_dept, nom) VALUES 
(1, 'Informatique'),
(2, 'Mathématiques'),
(3, 'Physique'),
(4, 'Chimie'),
(5, 'Biologie');

-- Insertion Enseignants
INSERT INTO Enseignant (id_enseignant, nom, prenom, id_dept, specialite) VALUES 
(1, 'Dupont', 'Jean', 1, 'Base de données'),
(2, 'Martin', 'Sophie', 1, 'Intelligence artificielle'),
(3, 'Bernard', 'Pierre', 2, 'Analyse numérique'),
(4, 'Thomas', 'Marie', 3, 'Mécanique quantique'),
(5, 'Robert', 'Claire', 4, 'Chimie organique');

-- Insertion Cours
INSERT INTO Cours (id_cours, nom_cours, id_enseignant, credit) VALUES 
(1, 'Introduction aux bases de données', 1, 5),
(2, 'Programmation en Python', 2, 4),
(3, 'Algèbre linéaire', 3, 6),
(4, 'Physique moderne', 4, 5),
(5, 'Chimie générale', 5, 4),
(6, 'Structures de données', 1, 5);

-- Insertion Étudiants
INSERT INTO Etudiant (id_etudiant, nom, prenom, date_naissance, id_dept) VALUES 
(1, 'Petit', 'Lucas', '2000-05-12', 1),
(2, 'Durand', 'Emma', '2001-09-25', 1),
(3, 'Moreau', 'Noah', '1999-11-30', 2),
(4, 'Lefebvre', 'Chloé', '2002-03-17', 3),
(5, 'Garcia', 'Louis', '2000-08-08', 4),
(6, 'Roux', 'Jade', '2001-01-22', 1),
(7, 'Fournier', 'Léo', '1999-07-14', 2);

-- Insertion Inscriptions
INSERT INTO Inscription (id_etudiant, id_cours, note) VALUES 
(1, 1, 17.5),
(1, 2, 16.0),
(2, 1, 14.5),
(2, 3, 15.0),
(3, 3, 18.0),
(3, 4, 16.5),
(4, 4, 15.5),
(4, 5, 14.0),
(5, 5, 13.5),
(5, 6, 16.5),
(6, 1, 15.0),
(6, 2, 17.0),
(7, 3, 14.5);

-- Insertion Salles
INSERT INTO Salle (id_salle, nom_salle, capacite) VALUES 
(1, 'A101', 30),
(2, 'B201', 50),
(3, 'C301', 100),
(4, 'D101', 25),
(5, 'E201', 40);

-- Insertion Séances
INSERT INTO Seance (id_seance, id_cours, id_salle, date_seance, heure_debut, heure_fin) VALUES 
(1, 1, 2, '2024-05-15', '08:00', '10:00'),
(2, 2, 1, '2024-05-15', '10:15', '12:15'),
(3, 3, 3, '2024-05-15', '13:30', '15:30'),
(4, 4, 4, '2024-05-16', '09:00', '11:00'),
(5, 5, 5, '2024-05-16', '13:00', '15:00'),
(6, 6, 2, '2024-05-17', '10:00', '12:00');

-- ============================================================================
-- 15 REQUÊTES SQL COMPLÈTES
-- ============================================================================

-- 1. REQUÊTE SANS JOINTURE AVEC WHERE #1 - Liste des étudiants du département d'informatique
SELECT 'Query 1: Étudiants en informatique' AS Description;
SELECT e.id_etudiant, e.nom, e.prenom 
FROM Etudiant e
WHERE e.id_dept = 1;

-- 2. REQUÊTE SANS JOINTURE AVEC WHERE #2 - Salles avec capacité supérieure à 30
SELECT 'Query 2: Salles de capacité > 30' AS Description;
SELECT id_salle, nom_salle, capacite
FROM Salle
WHERE capacite > 30;

-- 3. REQUÊTE AVEC JOINTURE #1 - Cours de plus de 4 crédits avec leurs enseignants
SELECT 'Query 3: Cours de plus de 4 crédits' AS Description;
SELECT c.id_cours, c.nom_cours, e.nom, e.prenom
FROM Cours c
JOIN Enseignant e ON c.id_enseignant = e.id_enseignant
WHERE c.credit > 4;

-- 4. REQUÊTE AVEC JOINTURE #2 - Étudiants ayant une note supérieure à 16
SELECT 'Query 4: Étudiants avec note > 16' AS Description;
SELECT e.nom, e.prenom, c.nom_cours, i.note
FROM Inscription i
JOIN Etudiant e ON i.id_etudiant = e.id_etudiant
JOIN Cours c ON i.id_cours = c.id_cours
WHERE i.note > 16;

-- 5. REQUÊTE AVEC JOINTURE GAUCHE #3 - Nombre d'étudiants par cours pour les cours du département d'informatique
SELECT 'Query 5: Étudiants par cours (informatique)' AS Description;
SELECT c.nom_cours, COUNT(i.id_etudiant) AS nombre_etudiants
FROM Cours c
LEFT JOIN Inscription i ON c.id_cours = i.id_cours
JOIN Enseignant e ON c.id_enseignant = e.id_enseignant
WHERE e.id_dept = 1
GROUP BY c.id_cours, c.nom_cours;

-- 6. REQUÊTE AVEC JOINTURE COMPLÈTE #4 - Jointure complète Salles/Séances
SELECT 'Query 6: Jointure complète Salles/Séances' AS Description;
SELECT 
    s.id_salle,
    s.nom_salle,
    se.id_seance,
    se.date_seance,
    se.heure_debut,
    se.heure_fin
FROM Salle s
LEFT JOIN Seance se ON s.id_salle = se.id_salle
UNION ALL
SELECT
    s.id_salle,
    s.nom_salle,
    se.id_seance,
    se.date_seance,
    se.heure_debut,
    se.heure_fin
FROM Seance se
LEFT JOIN Salle s ON s.id_salle = se.id_salle
WHERE s.id_salle IS NULL;

-- 7. REQUÊTE AVEC IN - Salles disponibles le 16 mai 2024 entre 10h et 12h
SELECT 'Query 7: Salles disponibles le 2024-05-16 10h-12h' AS Description;
SELECT s.id_salle, s.nom_salle
FROM Salle s
WHERE s.id_salle NOT IN (
    SELECT se.id_salle
    FROM Seance se
    WHERE se.date_seance = '2024-05-16'
    AND (
        (se.heure_debut <= '10:00' AND se.heure_fin > '10:00') OR
        (se.heure_debut < '12:00' AND se.heure_fin >= '12:00') OR
        (se.heure_debut >= '10:00' AND se.heure_fin <= '12:00')
    )
);

-- 8. REQUÊTE AVEC EXISTS - Enseignants qui ont au moins un cours
SELECT 'Query 8: Enseignants avec cours' AS Description;
SELECT e.nom, e.prenom, e.specialite
FROM Enseignant e
WHERE EXISTS (
    SELECT 1 FROM Cours c 
    WHERE c.id_enseignant = e.id_enseignant
);

-- 9. REQUÊTE AVEC REGROUPEMENT #1 - Moyenne des notes pour chaque cours
SELECT 'Query 9: Moyennes par cours' AS Description;
SELECT c.nom_cours, AVG(i.note) AS moyenne
FROM Inscription i
JOIN Cours c ON i.id_cours = c.id_cours
GROUP BY c.id_cours, c.nom_cours;

-- 10. REQUÊTE AVEC REGROUPEMENT ET HAVING #2 - Cours avec moyenne supérieure à 15
SELECT 'Query 10: Cours avec moyenne > 15' AS Description;
SELECT c.nom_cours, AVG(i.note) AS moyenne
FROM Inscription i
JOIN Cours c ON i.id_cours = c.id_cours
GROUP BY c.id_cours, c.nom_cours
HAVING AVG(i.note) > 15;

-- 11. REQUÊTE AVEC SOUS-REQUÊTE #1 - Meilleur étudiant par cours (note la plus élevée)
SELECT 'Query 11: Meilleurs étudiants par cours' AS Description;
SELECT c.nom_cours, e.nom, e.prenom, i.note
FROM Inscription i
JOIN Etudiant e ON i.id_etudiant = e.id_etudiant
JOIN Cours c ON i.id_cours = c.id_cours
WHERE (i.id_cours, i.note) IN (
    SELECT id_cours, MAX(note)
    FROM Inscription
    GROUP BY id_cours
);

-- 12. REQUÊTE AVEC SOUS-REQUÊTE #2 - Total des crédits par étudiant
SELECT 'Query 12: Crédits par étudiant' AS Description;
SELECT e.nom, e.prenom, 
       (SELECT SUM(c.credit) 
        FROM Inscription i2 
        JOIN Cours c ON i2.id_cours = c.id_cours 
        WHERE i2.id_etudiant = e.id_etudiant) AS total_credits
FROM Etudiant e
WHERE EXISTS (
    SELECT 1 FROM Inscription i WHERE i.id_etudiant = e.id_etudiant
);

-- 13. REQUÊTE AVEC ORDER BY - Enseignants et leur nombre de cours (triés par nombre décroissant)
SELECT 'Query 13: Enseignants par nombre de cours' AS Description;
SELECT e.nom, e.prenom, COUNT(c.id_cours) AS nombre_cours
FROM Enseignant e
LEFT JOIN Cours c ON e.id_enseignant = c.id_enseignant
GROUP BY e.id_enseignant, e.nom, e.prenom
ORDER BY nombre_cours DESC;

-- 14. REQUÊTE DE MODIFICATION - Augmenter la note de Lucas Petit en base de données
SELECT 'Query 14: Modification de note' AS Description;
SELECT 'Before update:' AS State, e.nom, e.prenom, c.nom_cours, i.note
FROM Inscription i
JOIN Etudiant e ON i.id_etudiant = e.id_etudiant
JOIN Cours c ON i.id_cours = c.id_cours
WHERE e.nom = 'Petit' AND e.prenom = 'Lucas' 
  AND c.nom_cours = 'Introduction aux bases de données';

UPDATE Inscription 
SET note = 18.0 
WHERE id_etudiant = (SELECT id_etudiant FROM Etudiant WHERE nom = 'Petit' AND prenom = 'Lucas')
AND id_cours = (SELECT id_cours FROM Cours WHERE nom_cours = 'Introduction aux bases de données');

SELECT 'After update:' AS State, e.nom, e.prenom, c.nom_cours, i.note
FROM Inscription i
JOIN Etudiant e ON i.id_etudiant = e.id_etudiant
JOIN Cours c ON i.id_cours = c.id_cours
WHERE e.nom = 'Petit' AND e.prenom = 'Lucas' 
  AND c.nom_cours = 'Introduction aux bases de données';

-- 15. REQUÊTE DE SUPPRESSION - Supprimer les inscriptions avec une note inférieure à 14
SELECT 'Query 15: Suppression des notes < 14' AS Description;
SELECT 'Before deletion:' AS State, COUNT(*) AS count_low_grades FROM Inscription WHERE note < 14;

DELETE FROM Inscription 
WHERE note < 14;

SELECT 'After deletion:' AS State, COUNT(*) AS count_low_grades FROM Inscription WHERE note < 14;

-- ============================================================================
-- TRIGGER TESTS
-- ============================================================================

SELECT '=== Trigger Tests ===' AS Description;

-- Test 1: Should fail (overlapping session)
SELECT 'Test 1: Insert overlapping session (should fail)' AS Test;
INSERT INTO Seance (id_seance, id_cours, id_salle, date_seance, heure_debut, heure_fin)
VALUES (7, 1, 2, '2024-05-15', '09:00', '11:00');

-- Test 2: Should succeed (non-overlapping session)
SELECT 'Test 2: Insert non-overlapping session (should succeed)' AS Test;
INSERT INTO Seance (id_seance, id_cours, id_salle, date_seance, heure_debut, heure_fin)
VALUES (7, 1, 2, '2024-05-15', '11:00', '13:00');

SELECT 'Trigger test results:' AS Description;
SELECT 'Test 1: ' || (CASE WHEN (SELECT COUNT(*) FROM Seance WHERE id_seance = 7) = 0 
                            THEN 'PASS' ELSE 'FAIL' END) AS Result;
SELECT 'Test 2: ' || (CASE WHEN (SELECT COUNT(*) FROM Seance WHERE id_seance = 7) = 1 
                            THEN 'PASS' ELSE 'FAIL' END) AS Result;

-- Cleanup after trigger tests
DELETE FROM Seance WHERE id_seance = 7;