import numpy as np

# FFT
def compute_rfft(S):
    """
    Computes the Real Fast Fourier Transform (RFFT) of a sinusoidal signal.

    Parameters:
        S (list or np.ndarray): Sinusoidal signal.

    Returns:
        A: Amplitud or Magnitud
        phi: Phase
        np.ndarray: NumPy array of complex numbers. Each complex number represents the magnitude and phase of a particular frequency component of the input signal.
    """
    if S is None:
        raise ValueError("Input signal is missing.")
    if not isinstance(S, (list, np.ndarray)):
        raise TypeError("Input signal must be a list or numpy.ndarray.")

    # Compute the RFFT
    rfft_result = np.fft.rfft(S)

    # Extract Amplitud and Phase
    A = np.abs(rfft_result) # M = sqrt(a^2 + b^2)
    phi = np.angle(rfft_result) # phi = arctan(b / a)
    return rfft_result, A, phi

def compute_irfft(S, factor):
    """
    Perform IRFFT variations on audio data based on the upsampling or downsampling factor.

    Parameters:
        S (list or np.ndarray): Sinusoidal signal.
        factor (float): Upsampling or downsampling factor.

    Returns:
        numpy.ndarray: IRFFT output with modified length based on the factor.
    """
    if S is None:
        raise ValueError("Input signal is missing.")
    if not isinstance(S, (list, np.ndarray)):
        raise TypeError("Input signal must be a list or numpy.ndarray.")
    # Compute the RFFT of the audio data using the compute_rfft function
    rfft_result, _, _ = compute_rfft(S)

    N=len(S)
    # Determine the length of the IRFFT output based on the factor
    if factor > 1:
        # Upsampling: Multiply the length of S by the factor
        length = int(N * factor)
    elif factor < 1:
        # Downsampling: Divide the length of S by the factor
        length = int(N / factor)
    else:
        # No modification if factor is 1
        return np.fft.irfft(rfft_result, N)

    # Apply IRFFT with the modified length
    return np.fft.irfft(rfft_result, length)

def compute_irfft_fixed_length(S, factor):
    """
    Modify frequencies of a signal by a given factor without changing the length of the signal.

    Parameters:
        S (list or np.ndarray): Sinusoidal signal.
        factor (float): Frequency modification factor.

    Returns:
        np.ndarray: The modified signal.
    """
    if S is None:
        raise ValueError("Input signal is missing.")
    if not isinstance(S, (list, np.ndarray)):
        raise TypeError("Input signal must be a list or numpy.ndarray.")
    if factor <= 0:
        raise ValueError("Factor must be a positive float.")

    # Compute the RFFT of the audio data
    rfft_result, _, _ = compute_rfft(S)

    N = len(rfft_result)
    # Create an empty array for the modified RFFT result
    modified_rfft = np.zeros_like(rfft_result, dtype=np.complex128)

    # Modify the frequencies by the factor
    for i in range(N):
        modified_index = int(i * factor)
        if modified_index < N:
            modified_rfft[modified_index] = rfft_result[i]

    # Compute the IRFFT with the original length
    modified_signal = np.fft.irfft(modified_rfft, len(S))

    return modified_signal
