import numpy as np

def generate_frequency_vector_f_fs(Fs, N):
    """
    Generate a frequency vector for a given sample rate and number of samples.

    Parameters:
        Fs (int or float): Sample rate in Hz.
        N (int): Number of samples.

    Returns:
        numpy.ndarray: Frequency vector.
    """
    if Fs is None or N is None:
        raise ValueError("Missing Sample rate (Fs) or Number of samples (N).")
    if not isinstance(Fs, (int, float)) or not isinstance(N, int):
        raise TypeError("Sample rate (Fs) must be int or float, and Number of samples (N) must be int.")
    fs = 1/Fs
    fr = np.fft.fftfreq(N, Fs)
    return fr,fs

def generate_f_and_n_for_lists_of_N(Fs, N):
    """
    Generate a frequency vector based on the sample rate and number of samples.

    Parameters:
        Fs (int or float): Sample rate in Hz (samples per second).
        N (int or list or numpy.ndarray): Number of samples for each signal.
    
    Returns:
        list: Frequency vectors for each number of samples.
        int: Length of the last frequency vector.
    """
    if Fs is None or N is None:
        raise ValueError("Missing sample rate (Fs) or number of samples (N).")
    if not isinstance(Fs, (int, float)):
        raise TypeError("Sample rate (Fs) must be an int or float.")
    if not isinstance(N, (list, np.ndarray, int)):
        raise TypeError("Number of samples (N) must be an int, list, or numpy.ndarray.")
    
    # If N is a single int, convert it to a list
    if isinstance(N, int):
        N = [N]
    
    # Generate the frequency vectors for each number of samples
    frequency_vectors = []
    for n in N:
        fs = 1 / Fs
        f = np.fft.fftfreq(n, fs)  # Generate the frequency vector
        frequency_vectors.append(f)
    
    return frequency_vectors, n

def generate_sine_wave_freq_domain(A, fr, N, Fs, phi=0):
    """
    Generate a frequency domain representation of a sinusoidal signal.

    Parameters:
        A (int or float): Amplitude of the sinusoid.
        fr (int or float): Frequency of the sinusoid (Hz).
        N (int): Number of samples.
        Fs (int or float): Sample rate in Hz.
        phi (int or float, optional): Phase of the sinusoid in radians. Defaults to 0.

    Returns:
        np.ndarray: Frequency domain representation of the sinusoidal signal.
    """
    if fr is None or A is None or N is None or Fs is None:
        raise ValueError("Missing amplitude (A), frequency (fr), number of samples (N), or sample rate (Fs).")
    if not isinstance(fr, (int, float)) or not isinstance(A, (int, float)) or not isinstance(Fs, (int, float)) or not isinstance(N, int):
        raise TypeError("Amplitude (A), frequency (fr), sample rate (Fs) must be int or float, and number of samples (N) must be int.")

    # Initialize frequency domain representation with zeros
    freq_domain = np.zeros(N, dtype=complex)
    
    # Calculate the index corresponding to the frequency
    k = int(fr * N / Fs)
    
    # Set the amplitude and phase for the positive frequency
    freq_domain[k] = A * np.exp(1j * phi)
    
    # Set the amplitude and phase for the negative frequency if k is not 0 or N/2
    if k != 0 and k != N // 2:
        freq_domain[N - k] = A * np.exp(-1j * phi)

    return freq_domain

def generate_sinusoidal_signal_freq_domain(A, freqs, phi, N, Fs):
    """
    Generate a frequency domain representation of a signal by superimposing sinusoids.

    Parameters:
        A (list): List of amplitudes for each sinusoid.
        freqs (list): List of frequencies for each sinusoid (Hz).
        phi (list): List of phases for each sinusoid (radians).
        N (int): Number of samples.
        Fs (int or float): Sample rate in Hz.

    Returns:
        np.ndarray: Frequency domain representation of the signal.
    """
    if len(A) != len(freqs) or len(A) != len(phi):
        raise ValueError("Lengths of amplitudes, frequencies, and phases must be the same.")

    # Initialize frequency domain representation with zeros
    freq_domain = np.zeros(N, dtype=complex)

    # Add each sinusoidal component
    for a, f, p in zip(A, freqs, phi):
        k = int(f * N / Fs)
        freq_domain[k] = a * np.exp(1j * p)
        if k != 0 and k != N // 2:
            freq_domain[N - k] = a * np.exp(-1j * p)
    
    return freq_domain

def generate_square_wave_freq_domain(Fs, T, N):
    """
    Generate a frequency domain representation of a square wave.

    Parameters:
        Fs (int): Sampling frequency (Hz).
        T (float): Duration of the signal (seconds).
        N (int): Number of samples.

    Returns:
        np.ndarray: Frequency domain representation of the square wave.
    """
    # Initialize frequency domain representation with zeros
    freq_domain = np.zeros(N, dtype=complex)
    
    # Number of harmonics to consider (Nyquist limit)
    num_harmonics = N // 2
    
    # Fundamental frequency
    f0 = 1 / T
    
    # Add odd harmonics
    for k in range(1, num_harmonics, 2):
        index = int(k * f0 * N / Fs)
        freq_domain[index] = 1 / k
        if index != 0 and index != N // 2:
            freq_domain[N - index] = 1 / k

    return freq_domain



