import numpy as np
from fft import compute_rfft

        
# TIME COMAIN SIGNAL PROCESSING
def generate_time_vector_t_fs_n(Fs ,T, start_time=0):
    """
    Generate a time vector based on the sample rate and duration of the signal.

    Parameters:
        Fs (int or float): Sample rate in Hz per second (Fréquence d'échantillonnage).
        T (int or float): Duration of the signal in seconds (Durée de l'échantillonage).
        start_time (int or float, optional): Start time of the signal. Defaults to 0.
    
    transformation:
         fs:       1/fs

    Returns:
        numpy.ndarray: Time vector representing the sampled time instances.
    """
    if Fs is None or T is None:
        raise ValueError("Missing Sample rate (fs) or Duration (T).")
    if not isinstance(Fs, (int, float)) or not isinstance(T, (int, float)):
        raise TypeError("Sample rate (fs) and Duration (T) must be int or float.")
    
    fs = 1 / Fs
    n = int(T * fs)  # Calculate the number of samples based on the duration and sample rate
    t = np.linspace(start_time, T, n, endpoint=False)  # Use linspace to generate the time vector
    return t, fs, n

def generate_t_and_n_for_lists_of_T(Fs, T, start_time=0):
    """
    Generate a time vector based on the sample rate and duration of the signal.

    Parameters:
        Fs (int or float): Sample rate in Hz (samples per second).
        T (float or list or numpy.ndarray): Duration(s) of the signal in seconds.
        start_time (int or float, optional): Start time of the signal. Defaults to 0.
    
    Returns:
        N(Number of Samples): lt
        numpy.ndarray: Time vector representing the sampled time instances.
    """
    if Fs is None or T is None:
        raise ValueError("Missing sample rate (fs) or duration (T).")
    if not isinstance(Fs, (int, float)):
        raise TypeError("Sample rate (fs) must be an int or float.")
    if not isinstance(T, (list, np.ndarray, float)):
        raise TypeError("Duration (T) must be a float, list, or numpy.ndarray.")
    
    # If T is a single float, convert it to a list
    if isinstance(T, float):
        T = [T]
    
    # Generate the time vectors for each duration
    time_vectors = []
    for T in T:
        n = int(T * Fs) # at this point fs is actually fs !!!!
        t = np.linspace(start_time, T, n, endpoint=False)  # Exclude the end value
        #t_vec=np.array(time_vectors.append(t))
        #t_vec = time_vectors.append(t)
    return t, n

def generate_sine_wave(A, fr, t, phi=0):
    """
    Generate a sinusoidal signal based on the time vector.

    Parameters:
        A (int or float): Amplitude, the maximum displacement of the waveform from its mean value.
        fr (int or float): Fundamental Frequency, oscillations per unit of time (Hz).
        t (list or np.ndarray): Time vector.

    Returns:
        np.ndarray: NumPy array containing the calculated values of the sinusoidal waveform at each time point specified by t.
    """
    if fr is None or A is None or t is None:
        raise ValueError("Missing amplitude (A), frequency (fr), or time vector (t).")
    if not isinstance(fr, (int, float)) or not isinstance(A, (int, float)):
        raise TypeError("Amplitude (A) and frequency (fr) must be int or float.")
    if not isinstance(t, (list, np.ndarray)):
        raise TypeError("Time vector (t) must be a list or numpy.ndarray.")

    t = np.array(t)  # Ensure time vector is a NumPy array for element-wise operations
    S = A * np.sin(2 * np.pi * fr * t + phi) #S(t)=A*sin(2π*fr*​t+phi) here a phi is radian(ex. np.pi/2)
    return S

def generate_sinusiodal_signal(A, freqs, phi, t):
    """
    Generate a signal by superimposing sinusoids with given parameters.

    Parameters:
        amplitudes (list): List of amplitudes for each sinusoid.
        frequencies (list): List of frequencies for each sinusoid (Hz).
        phases (list): List of phases for each sinusoid (radians).
        t (numpy.ndarray): Time vector.

    Returns:
        numpy.ndarray: Signal as a list of 16-bit integers.
    """
    # Check if the lengths of input lists match
    if len(A) != len(freqs) or len(A): #!= len(phi):
        raise ValueError("Lengths of amplitudes, frequencies, and phases must be the same.")
    
    S_list = np.zeros(freqs, dtype=complex)
    
    # Create the signal by superimposing sinusoids
    signal = sum(a * np.sin(2 * np.pi * f * t + p) for a, f, p in zip(A, freqs, phi)) #S(t)=A0 + sum(A(n)*(sin(2π*fr(n)*​t+phi(n))) 

    # Convert the signal to a list of 16-bit integers
    S_list = np.int16(signal)

    return S_list

def generate_squared_signal(Fs, T, t, center_range=0.5):
    """
    Generate a squared signal.

    Parameters:
        Fs (int): Sampling frequency (Hz).
        T (float): Duration of the signal (seconds).
        start_time (float, optional): Start time of the signal (seconds). Defaults to 0.

    Returns:
        list: Squared signal.
    """
    # Calculate the number of samples
    n = int(T * Fs)

    fs = 1/Fs

    Mt = t % fs

    # Generate the squared signal
    square_S = np.sign(2 * np.pi * Mt - center_range) #sign returns the sign of each element

    return square_S


